/**
 * Get the primary MIME type for a given file extension.
 *
 * @param {string} extension - The file extension (with or without leading dot).
 * @returns {string} The first associated MIME type, or 'application/octet-stream' if unknown.
 */
function getMimeType(extension) {
	const mimeTypes = {
		'123': ['application/lotus123', 'application/vnd.lotus-1-2-3', 'application/wk1', 'application/x-123', 'application/x-lotus123', 'zz-application/zz-winassoc-123'],
		'1km': ['application/vnd.1000minds.decision-model+xml'],
		'32x': ['application/x-genesis-32x-rom'],
		'3dml': ['text/vnd.in3d.3dml'],
		'3ds': ['application/x-nintendo-3ds-rom', 'image/x-3ds'],
		'3dsx': ['application/x-nintendo-3ds-executable'],
		'3g2': ['audio/3gpp2', 'video/3gpp2'],
		'3ga': ['audio/3gpp', 'audio/3gpp-encrypted', 'audio/x-rn-3gpp-amr', 'audio/x-rn-3gpp-amr-encrypted', 'audio/x-rn-3gpp-amr-wb', 'audio/x-rn-3gpp-amr-wb-encrypted', 'video/3gp', 'video/3gpp', 'video/3gpp-encrypted'],
		'3gp': ['audio/3gpp', 'audio/3gpp-encrypted', 'audio/x-rn-3gpp-amr', 'audio/x-rn-3gpp-amr-encrypted', 'audio/x-rn-3gpp-amr-wb', 'audio/x-rn-3gpp-amr-wb-encrypted', 'video/3gp', 'video/3gpp', 'video/3gpp-encrypted'],
		'3gp2': ['audio/3gpp2', 'video/3gpp2'],
		'3gpp': ['audio/3gpp', 'audio/3gpp-encrypted', 'audio/x-rn-3gpp-amr', 'audio/x-rn-3gpp-amr-encrypted', 'audio/x-rn-3gpp-amr-wb', 'audio/x-rn-3gpp-amr-wb-encrypted', 'video/3gp', 'video/3gpp', 'video/3gpp-encrypted'],
		'3gpp2': ['audio/3gpp2', 'video/3gpp2'],
		'3mf': ['application/vnd.ms-3mfdocument', 'model/3mf'],
		'602': ['application/x-t602'],
		'669': ['audio/x-mod'],
		'7z': ['application/x-7z-compressed'],
		'7z.001': ['application/x-7z-compressed'],
		'C': ['text/x-c++src'],
		'PAR2': ['application/x-par2'],
		'PL': ['application/x-perl', 'text/x-perl'],
		'Z': ['application/x-compress'],
		'a': ['application/x-archive'],
		'a26': ['application/x-atari-2600-rom'],
		'a78': ['application/x-atari-7800-rom'],
		'aa': ['audio/vnd.audible', 'audio/x-pn-audibleaudio'],
		'aab': ['application/x-authorware-bin'],
		'aac': ['audio/aac', 'audio/x-aac', 'audio/x-hx-aac-adts'],
		'aam': ['application/x-authorware-map'],
		'aas': ['application/x-authorware-seg'],
		'aax': ['audio/vnd.audible', 'audio/vnd.audible.aax', 'audio/x-pn-audibleaudio'],
		'aaxc': ['audio/vnd.audible.aaxc'],
		'abw': ['application/x-abiword'],
		'abw.CRASHED': ['application/x-abiword'],
		'abw.gz': ['application/x-abiword'],
		'ac': ['application/pkix-attr-cert', 'application/vnd.nokia.n-gage.ac+xml'],
		'ac3': ['audio/ac3'],
		'acc': ['application/vnd.americandynamics.acc'],
		'ace': ['application/x-ace', 'application/x-ace-compressed'],
		'acu': ['application/vnd.acucobol'],
		'acutc': ['application/vnd.acucorp'],
		'adb': ['text/x-adasrc'],
		'adf': ['application/x-amiga-disk-format'],
		'adp': ['audio/adpcm'],
		'ads': ['text/x-adasrc'],
		'adts': ['audio/aac', 'audio/x-aac', 'audio/x-hx-aac-adts'],
		'aep': ['application/vnd.audiograph'],
		'afm': ['application/x-font-afm', 'application/x-font-type1'],
		'afp': ['application/vnd.ibm.modcap'],
		'ag': ['image/x-applix-graphics'],
		'agb': ['application/x-gba-rom'],
		'age': ['application/vnd.age'],
		'ahead': ['application/vnd.ahead.space'],
		'ai': ['application/illustrator', 'application/postscript', 'application/vnd.adobe.illustrator'],
		'aif': ['audio/x-aiff'],
		'aifc': ['audio/x-aifc', 'audio/x-aiff', 'audio/x-aiffc'],
		'aiff': ['audio/x-aiff'],
		'aiffc': ['audio/x-aifc', 'audio/x-aiffc'],
		'air': ['application/vnd.adobe.air-application-installer-package+zip'],
		'ait': ['application/vnd.dvb.ait'],
		'al': ['application/x-perl', 'text/x-perl'],
		'alz': ['application/x-alz'],
		'ami': ['application/vnd.amiga.ami'],
		'aml': ['application/automationml-aml+xml'],
		'amlx': ['application/automationml-amlx+zip'],
		'amr': ['audio/amr', 'audio/amr-encrypted'],
		'amz': ['audio/x-amzxml'],
		'ani': ['application/x-navi-animation'],
		'anim1': ['video/x-anim'],
		'anim2': ['video/x-anim'],
		'anim3': ['video/x-anim'],
		'anim4': ['video/x-anim'],
		'anim5': ['video/x-anim'],
		'anim6': ['video/x-anim'],
		'anim7': ['video/x-anim'],
		'anim8': ['video/x-anim'],
		'anim9': ['video/x-anim'],
		'animj': ['video/x-anim'],
		'anx': ['application/annodex', 'application/x-annodex'],
		'ape': ['audio/x-ape'],
		'apk': ['application/vnd.android.package-archive'],
		'apng': ['image/apng', 'image/vnd.mozilla.apng'],
		'appcache': ['text/cache-manifest'],
		'appimage': ['application/vnd.appimage', 'application/x-iso9660-appimage'],
		'appinstaller': ['application/appinstaller'],
		'application': ['application/x-ms-application'],
		'appx': ['application/appx'],
		'appxbundle': ['application/appxbundle'],
		'apr': ['application/vnd.lotus-approach'],
		'ar': ['application/x-archive'],
		'arc': ['application/x-freearc'],
		'arj': ['application/x-arj'],
		'arw': ['image/x-sony-arw'],
		'as': ['application/x-applix-spreadsheet'],
		'asar': ['application/x-asar'],
		'asc': ['application/pgp', 'application/pgp-encrypted', 'application/pgp-keys', 'application/pgp-signature', 'text/plain'],
		'asd': ['text/x-common-lisp'],
		'asf': ['application/vnd.ms-asf', 'video/x-ms-asf', 'video/x-ms-asf-plugin', 'video/x-ms-wm'],
		'asice': ['application/vnd.etsi.asic-e+zip'],
		'asm': ['text/x-asm'],
		'aso': ['application/vnd.accpac.simply.aso'],
		'asp': ['application/x-asp'],
		'ass': ['audio/aac', 'audio/x-aac', 'audio/x-hx-aac-adts', 'text/x-ssa'],
		'astc': ['image/astc'],
		'asx': ['application/x-ms-asx', 'audio/x-ms-asx', 'video/x-ms-asf', 'video/x-ms-wax', 'video/x-ms-wmx', 'video/x-ms-wvx'],
		'atc': ['application/vnd.acucorp'],
		'atom': ['application/atom+xml'],
		'atomcat': ['application/atomcat+xml'],
		'atomdeleted': ['application/atomdeleted+xml'],
		'atomsvc': ['application/atomsvc+xml'],
		'atx': ['application/vnd.antix.game-component'],
		'au': ['audio/basic'],
		'automount': ['text/x-systemd-unit'],
		'avci': ['image/avci'],
		'avcs': ['image/avcs'],
		'avf': ['video/avi', 'video/divx', 'video/msvideo', 'video/vnd.avi', 'video/vnd.divx', 'video/x-avi', 'video/x-msvideo'],
		'avi': ['video/avi', 'video/divx', 'video/msvideo', 'video/vnd.avi', 'video/vnd.divx', 'video/x-avi', 'video/x-msvideo'],
		'avif': ['image/avif', 'image/avif-sequence'],
		'avifs': ['image/avif', 'image/avif-sequence'],
		'aw': ['application/applixware', 'application/x-applix-word'],
		'awb': ['audio/amr-wb', 'audio/amr-wb-encrypted'],
		'awk': ['application/x-awk'],
		'axa': ['audio/annodex', 'audio/x-annodex'],
		'axv': ['video/annodex', 'video/x-annodex'],
		'azf': ['application/vnd.airzip.filesecure.azf'],
		'azs': ['application/vnd.airzip.filesecure.azs'],
		'azv': ['image/vnd.airzip.accelerator.azv'],
		'azw': ['application/vnd.amazon.ebook'],
		'azw3': ['application/vnd.amazon.mobi8-ebook', 'application/x-mobi8-ebook'],
		'b16': ['image/vnd.pco.b16'],
		'bak': ['application/x-trash'],
		'bary': ['model/vnd.bary'],
		'bas': ['text/x-basic'],
		'bat': ['application/bat', 'application/x-bat', 'application/x-msdownload'],
		'bcpio': ['application/x-bcpio'],
		'bdf': ['application/x-font-bdf'],
		'bdm': ['application/vnd.syncml.dm+wbxml', 'video/mp2t'],
		'bdmv': ['video/mp2t'],
		'bdo': ['application/vnd.nato.bindingdataobject+xml'],
		'bdoc': ['application/bdoc', 'application/x-bdoc'],
		'bed': ['application/vnd.realvnc.bed'],
		'bh2': ['application/vnd.fujitsu.oasysprs'],
		'bib': ['text/x-bibtex'],
		'bik': ['video/vnd.radgamettools.bink'],
		'bin': ['application/octet-stream'],
		'bk2': ['video/vnd.radgamettools.bink'],
		'blb': ['application/x-blorb'],
		'blend': ['application/x-blender'],
		'blender': ['application/x-blender'],
		'blorb': ['application/x-blorb'],
		'blp': ['text/x-blueprint'],
		'bmi': ['application/vnd.bmi'],
		'bmml': ['application/vnd.balsamiq.bmml+xml'],
		'bmp': ['image/bmp', 'image/x-bmp', 'image/x-ms-bmp'],
		'book': ['application/vnd.framemaker'],
		'box': ['application/vnd.previewsystems.box'],
		'boz': ['application/x-bzip2'],
		'bps': ['application/x-bps-patch'],
		'brk': ['chemical/x-pdb'],
		'bsdiff': ['application/x-bsdiff'],
		'bsp': ['model/vnd.valve.source.compiled-map'],
		'btf': ['image/prs.btif'],
		'btif': ['image/prs.btif'],
		'bz': ['application/bzip2', 'application/x-bzip', 'application/x-bzip1'],
		'bz2': ['application/x-bz2', 'application/bzip2', 'application/x-bzip', 'application/x-bzip2'],
		'bz3': ['application/x-bzip3'],
		'c': ['text/x-c', 'text/x-csrc'],
		'c++': ['text/x-c++src'],
		'c11amc': ['application/vnd.cluetrust.cartomobile-config'],
		'c11amz': ['application/vnd.cluetrust.cartomobile-config-pkg'],
		'c4d': ['application/vnd.clonk.c4group'],
		'c4f': ['application/vnd.clonk.c4group'],
		'c4g': ['application/vnd.clonk.c4group'],
		'c4p': ['application/vnd.clonk.c4group'],
		'c4u': ['application/vnd.clonk.c4group'],
		'cab': ['application/vnd.ms-cab-compressed', 'zz-application/zz-winassoc-cab'],
		'caf': ['audio/x-caf'],
		'cap': ['application/pcap', 'application/vnd.tcpdump.pcap', 'application/x-pcap'],
		'car': ['application/vnd.curl.car'],
		'cat': ['application/vnd.ms-pki.seccat'],
		'cb7': ['application/x-cb7', 'application/x-cbr'],
		'cba': ['application/x-cbr'],
		'cbl': ['text/x-cobol'],
		'cbor': ['application/cbor'],
		'cbr': ['application/vnd.comicbook-rar', 'application/x-cbr'],
		'cbt': ['application/x-cbr', 'application/x-cbt'],
		'cbz': ['application/vnd.comicbook+zip', 'application/x-cbr', 'application/x-cbz'],
		'cc': ['text/x-c', 'text/x-c++src'],
		'cci': ['application/x-nintendo-3ds-rom'],
		'ccmx': ['application/x-ccmx'],
		'cco': ['application/x-cocoa'],
		'cct': ['application/x-director'],
		'ccxml': ['application/ccxml+xml'],
		'cdbcmsg': ['application/vnd.contact.cmsg'],
		'cdf': ['application/x-netcdf'],
		'cdfx': ['application/cdfx+xml'],
		'cdi': ['application/x-discjuggler-cd-image'],
		'cdkey': ['application/vnd.mediastation.cdkey'],
		'cdmia': ['application/cdmi-capability'],
		'cdmic': ['application/cdmi-container'],
		'cdmid': ['application/cdmi-domain'],
		'cdmio': ['application/cdmi-object'],
		'cdmiq': ['application/cdmi-queue'],
		'cdr': ['application/cdr', 'application/coreldraw', 'application/vnd.corel-draw', 'application/x-cdr', 'application/x-coreldraw', 'image/cdr', 'image/x-cdr', 'zz-application/zz-winassoc-cdr'],
		'cdx': ['chemical/x-cdx'],
		'cdxml': ['application/vnd.chemdraw+xml'],
		'cdy': ['application/vnd.cinderella'],
		'cel': ['image/x-kiss-cel'],
		'cer': ['application/pkix-cert'],
		'cert': ['application/x-x509-ca-cert'],
		'cfs': ['application/x-cfs-compressed'],
		'cgb': ['application/x-gameboy-color-rom'],
		'cgm': ['image/cgm'],
		'chat': ['application/x-chat'],
		'chd': ['application/x-mame-chd'],
		'chm': ['application/vnd.ms-htmlhelp', 'application/x-chm'],
		'chrt': ['application/vnd.kde.kchart', 'application/x-kchart'],
		'cif': ['chemical/x-cif'],
		'cii': ['application/vnd.anser-web-certificate-issue-initiation'],
		'cil': ['application/vnd.ms-artgalry'],
		'cjs': ['application/node'],
		'cl': ['text/x-opencl-src'],
		'cla': ['application/vnd.claymore'],
		'class': ['application/java', 'application/java-byte-code', 'application/java-vm', 'application/x-java', 'application/x-java-class', 'application/x-java-vm'],
		'cld': ['model/vnd.cld'],
		'clkk': ['application/vnd.crick.clicker.keyboard'],
		'clkp': ['application/vnd.crick.clicker.palette'],
		'clkt': ['application/vnd.crick.clicker.template'],
		'clkw': ['application/vnd.crick.clicker.wordbank'],
		'clkx': ['application/vnd.crick.clicker'],
		'clp': ['application/x-msclip'],
		'clpi': ['video/mp2t'],
		'cls': ['application/x-tex', 'text/x-tex'],
		'cmake': ['text/x-cmake'],
		'cmc': ['application/vnd.cosmocaller'],
		'cmdf': ['chemical/x-cmdf'],
		'cml': ['chemical/x-cml'],
		'cmp': ['application/vnd.yellowriver-custom-menu'],
		'cmx': ['image/x-cmx'],
		'cob': ['text/x-cobol'],
		'cod': ['application/vnd.rim.cod'],
		'coffee': ['application/vnd.coffeescript', 'text/coffeescript'],
		'com': ['application/x-msdownload'],
		'conf': ['text/plain'],
		'cpi': ['video/mp2t'],
		'cpio': ['application/x-cpio'],
		'cpio.gz': ['application/x-cpio-compressed'],
		'cpl': ['application/cpl+xml', 'application/vnd.microsoft.portable-executable', 'application/x-ms-dos-executable', 'application/x-ms-ne-executable', 'application/x-msdownload'],
		'cpp': ['text/x-c', 'text/x-c++src'],
		'cpt': ['application/mac-compactpro'],
		'cr': ['text/crystal', 'text/x-crystal'],
		'cr2': ['image/x-canon-cr2'],
		'cr3': ['image/x-canon-cr3'],
		'crd': ['application/x-mscardfile'],
		'crdownload': ['application/x-partial-download'],
		'crl': ['application/pkix-crl'],
		'crt': ['application/x-x509-ca-cert'],
		'crw': ['image/x-canon-crw'],
		'crx': ['application/x-chrome-extension'],
		'cryptonote': ['application/vnd.rig.cryptonote'],
		'cs': ['text/x-csharp'],
		'csh': ['application/x-csh'],
		'csl': ['application/vnd.citationstyles.style+xml'],
		'csml': ['chemical/x-csml'],
		'cso': ['application/x-compressed-iso'],
		'csp': ['application/vnd.commonspace'],
		'css': ['text/css'],
		'cst': ['application/x-director'],
		'csv': ['text/csv', 'application/csv', 'text/x-comma-separated-values', 'text/x-csv'],
		'csvs': ['text/csv-schema'],
		'cu': ['application/cu-seeme'],
		'cue': ['application/x-cue'],
		'cur': ['image/x-win-bitmap'],
		'curl': ['text/vnd.curl'],
		'cwk': ['application/x-appleworks-document'],
		'cwl': ['application/cwl'],
		'cww': ['application/prs.cww'],
		'cxt': ['application/x-director'],
		'cxx': ['text/x-c', 'text/x-c++src'],
		'd': ['text/x-dsrc'],
		'dae': ['model/vnd.collada+xml'],
		'daf': ['application/vnd.mobius.daf'],
		'dar': ['application/x-dar'],
		'dart': ['application/vnd.dart', 'text/x-dart'],
		'dataless': ['application/vnd.fdsn.seed'],
		'davmount': ['application/davmount+xml'],
		'dbf': ['application/dbase', 'application/dbf', 'application/vnd.dbf', 'application/x-dbase', 'application/x-dbf'],
		'dbk': ['application/docbook+xml', 'application/vnd.oasis.docbook+xml', 'application/x-docbook+xml'],
		'dc': ['application/x-dc-rom'],
		'dcl': ['text/x-dcl'],
		'dcm': ['application/dicom'],
		'dcr': ['application/x-director', 'image/x-kodak-dcr'],
		'dcurl': ['text/vnd.curl.dcurl'],
		'dd2': ['application/vnd.oma.dd2+xml'],
		'ddd': ['application/vnd.fujixerox.ddd'],
		'ddf': ['application/vnd.syncml.dmddf+xml'],
		'dds': ['image/vnd.ms-dds', 'image/x-dds'],
		'deb': ['application/vnd.debian.binary-package', 'application/x-deb', 'application/x-debian-package'],
		'def': ['text/plain'],
		'der': ['application/x-x509-ca-cert'],
		'desktop': ['application/x-desktop', 'application/x-gnome-app-info'],
		'device': ['text/x-systemd-unit'],
		'dfac': ['application/vnd.dreamfactory'],
		'dff': ['audio/dff', 'audio/x-dff'],
		'dgc': ['application/x-dgc-compressed'],
		'di': ['text/x-dsrc'],
		'dia': ['application/x-dia-diagram'],
		'dib': ['image/bmp', 'image/x-bmp', 'image/x-ms-bmp'],
		'dic': ['text/x-c'],
		'diff': ['text/x-diff', 'text/x-patch'],
		'dir': ['application/x-director'],
		'dis': ['application/vnd.mobius.dis'],
		'disposition-notification': ['message/disposition-notification'],
		'divx': ['video/avi', 'video/divx', 'video/msvideo', 'video/vnd.avi', 'video/vnd.divx', 'video/x-avi', 'video/x-msvideo'],
		'djv': ['image/vnd.djvu', 'image/vnd.djvu+multipage', 'image/x-djvu', 'image/x.djvu'],
		'djvu': ['image/vnd.djvu', 'image/vnd.djvu+multipage', 'image/x-djvu', 'image/x.djvu'],
		'dll': ['application/vnd.microsoft.portable-executable', 'application/x-ms-dos-executable', 'application/x-ms-ne-executable', 'application/x-msdownload'],
		'dmg': ['application/x-apple-diskimage'],
		'dmp': ['application/pcap', 'application/vnd.tcpdump.pcap', 'application/x-pcap'],
		'dna': ['application/vnd.dna'],
		'dng': ['image/x-adobe-dng'],
		'doc': ['application/msword', 'application/vnd.ms-word', 'application/x-msword', 'zz-application/zz-winassoc-doc'],
		'docbook': ['application/docbook+xml', 'application/vnd.oasis.docbook+xml', 'application/x-docbook+xml'],
		'docm': ['application/vnd.ms-word.document.macroenabled.12'],
		'docx': ['application/vnd.openxmlformats-officedocument.wordprocessingml.document'],
		'dot': ['application/msword', 'application/msword-template', 'text/vnd.graphviz'],
		'dotm': ['application/vnd.ms-word.template.macroenabled.12'],
		'dotx': ['application/vnd.openxmlformats-officedocument.wordprocessingml.template'],
		'dp': ['application/vnd.osgi.dp'],
		'dpg': ['application/vnd.dpgraph'],
		'dpx': ['image/dpx'],
		'dra': ['audio/vnd.dra'],
		'drl': ['application/x-excellon'],
		'drle': ['image/dicom-rle'],
		'drv': ['application/vnd.microsoft.portable-executable', 'application/x-ms-dos-executable', 'application/x-ms-ne-executable', 'application/x-msdownload'],
		'dsc': ['text/prs.lines.tag'],
		'dsf': ['audio/dsd', 'audio/dsf', 'audio/x-dsd', 'audio/x-dsf'],
		'dsl': ['text/x-dsl'],
		'dssc': ['application/dssc+der'],
		'dtb': ['application/x-dtbook+xml', 'text/x-devicetree-binary'],
		'dtd': ['application/xml-dtd', 'text/x-dtd'],
		'dts': ['audio/vnd.dts', 'audio/x-dts', 'text/x-devicetree-source'],
		'dtshd': ['audio/vnd.dts.hd', 'audio/x-dtshd'],
		'dtsi': ['text/x-devicetree-source'],
		'dtx': ['application/x-tex', 'text/x-tex'],
		'dv': ['video/dv'],
		'dvb': ['video/vnd.dvb.file'],
		'dvi': ['application/x-dvi'],
		'dvi.bz2': ['application/x-bzdvi'],
		'dvi.gz': ['application/x-gzdvi'],
		'dwd': ['application/atsc-dwd+xml'],
		'dwf': ['model/vnd.dwf'],
		'dwg': ['image/vnd.dwg'],
		'dxf': ['image/vnd.dxf'],
		'dxp': ['application/vnd.spotfire.dxp'],
		'dxr': ['application/x-director'],
		'e': ['text/x-eiffel'],
		'ear': ['application/java-archive'],
		'ecelp4800': ['audio/vnd.nuera.ecelp4800'],
		'ecelp7470': ['audio/vnd.nuera.ecelp7470'],
		'ecelp9600': ['audio/vnd.nuera.ecelp9600'],
		'ecma': ['application/ecmascript'],
		'edm': ['application/vnd.novadigm.edm'],
		'edx': ['application/vnd.novadigm.edx'],
		'efi': ['application/vnd.microsoft.portable-executable'],
		'efif': ['application/vnd.picsel'],
		'egon': ['application/x-egon'],
		'ei6': ['application/vnd.pg.osasli'],
		'eif': ['text/x-eiffel'],
		'el': ['text/x-emacs-lisp'],
		'emf': ['application/emf', 'application/x-emf', 'application/x-msmetafile', 'image/emf', 'image/x-emf'],
		'eml': ['message/rfc822'],
		'emma': ['application/emma+xml'],
		'emotionml': ['application/emotionml+xml'],
		'emp': ['application/vnd.emusic-emusic_package'],
		'emz': ['application/x-msmetafile'],
		'ent': ['application/xml-external-parsed-entity', 'text/xml-external-parsed-entity'],
		'eol': ['audio/vnd.digital-winds'],
		'eot': ['application/vnd.ms-fontobject'],
		'eps': ['application/postscript', 'image/x-eps'],
		'eps.bz2': ['image/x-bzeps'],
		'eps.gz': ['image/x-gzeps'],
		'epsf': ['image/x-eps'],
		'epsf.bz2': ['image/x-bzeps'],
		'epsf.gz': ['image/x-gzeps'],
		'epsi': ['image/x-eps'],
		'epsi.bz2': ['image/x-bzeps'],
		'epsi.gz': ['image/x-gzeps'],
		'epub': ['application/epub+zip'],
		'eris': ['application/x-eris-link+cbor'],
		'erl': ['text/x-erlang'],
		'es': ['application/ecmascript', 'text/ecmascript'],
		'es3': ['application/vnd.eszigno3+xml'],
		'esa': ['application/vnd.osgi.subsystem'],
		'escn': ['application/x-godot-scene'],
		'esf': ['application/vnd.epson.esf'],
		'et3': ['application/vnd.eszigno3+xml'],
		'etheme': ['application/x-e-theme'],
		'etx': ['text/x-setext'],
		'eva': ['application/x-eva'],
		'evy': ['application/x-envoy'],
		'ex': ['text/x-elixir'],
		'exe': ['application/vnd.microsoft.portable-executable', 'application/x-dosexec', 'application/x-ms-dos-executable', 'application/x-ms-ne-executable', 'application/x-msdos-program', 'application/x-msdownload'],
		'exi': ['application/exi'],
		'exp': ['application/express'],
		'exr': ['image/aces', 'image/x-exr'],
		'exs': ['text/x-elixir'],
		'ext': ['application/vnd.novadigm.ext'],
		'ez': ['application/andrew-inset'],
		'ez2': ['application/vnd.ezpix-album'],
		'ez3': ['application/vnd.ezpix-package'],
		'f': ['text/x-fortran'],
		'f4a': ['audio/m4a', 'audio/mp4', 'audio/x-m4a'],
		'f4b': ['audio/x-m4b'],
		'f4v': ['video/mp4', 'video/mp4v-es', 'video/x-f4v', 'video/x-m4v'],
		'f77': ['text/x-fortran'],
		'f90': ['text/x-fortran'],
		'f95': ['text/x-fortran'],
		'fasl': ['text/x-common-lisp'],
		'fb2': ['application/x-fictionbook', 'application/x-fictionbook+xml'],
		'fb2.zip': ['application/x-zip-compressed-fb2'],
		'fbs': ['image/vnd.fastbidsheet'],
		'fcdt': ['application/vnd.adobe.formscentral.fcdt'],
		'fcs': ['application/vnd.isac.fcs'],
		'fd': ['application/x-fd-file', 'application/x-raw-floppy-disk-image'],
		'fdf': ['application/fdf', 'application/vnd.fdf'],
		'fds': ['application/x-fds-disk'],
		'fdt': ['application/fdt+xml'],
		'fe_launch': ['application/vnd.denovo.fcselayout-link'],
		'feature': ['text/x-gherkin'],
		'fg5': ['application/vnd.fujitsu.oasysgp'],
		'fgd': ['application/x-director'],
		'fh': ['image/x-freehand'],
		'fh4': ['image/x-freehand'],
		'fh5': ['image/x-freehand'],
		'fh7': ['image/x-freehand'],
		'fhc': ['image/x-freehand'],
		'fig': ['application/x-xfig', 'image/x-xfig'],
		'fish': ['application/x-fishscript', 'text/x-fish'],
		'fit': ['application/fits', 'image/fits', 'image/x-fits'],
		'fits': ['application/fits', 'image/fits', 'image/x-fits'],
		'fl': ['application/x-fluid'],
		'flac': ['audio/flac', 'audio/x-flac'],
		'flatpak': ['application/vnd.flatpak', 'application/vnd.xdgapp'],
		'flatpakref': ['application/vnd.flatpak.ref'],
		'flatpakrepo': ['application/vnd.flatpak.repo'],
		'flc': ['video/fli', 'video/x-fli', 'video/x-flic'],
		'fli': ['video/fli', 'video/x-fli', 'video/x-flic'],
		'flo': ['application/vnd.micrografx.flo'],
		'flv': ['video/x-flv', 'application/x-flash-video', 'flv-application/octet-stream', 'video/flv'],
		'flw': ['application/vnd.kde.kivio', 'application/x-kivio'],
		'flx': ['text/vnd.fmi.flexstor'],
		'fly': ['text/vnd.fly'],
		'fm': ['application/vnd.framemaker', 'application/x-frame'],
		'fnc': ['application/vnd.frogans.fnc'],
		'fo': ['application/vnd.software602.filler.form+xml', 'text/x-xslfo'],
		'fodg': ['application/vnd.oasis.opendocument.graphics-flat-xml'],
		'fodp': ['application/vnd.oasis.opendocument.presentation-flat-xml'],
		'fods': ['application/vnd.oasis.opendocument.spreadsheet-flat-xml'],
		'fodt': ['application/vnd.oasis.opendocument.text-flat-xml'],
		'for': ['text/x-fortran'],
		'fpx': ['image/vnd.fpx', 'image/x-fpx'],
		'frame': ['application/vnd.framemaker'],
		'fsc': ['application/vnd.fsc.weblaunch'],
		'fst': ['image/vnd.fst'],
		'ftc': ['application/vnd.fluxtime.clip'],
		'fti': ['application/vnd.anser-web-funds-transfer-initiation'],
		'fts': ['application/fits', 'image/fits', 'image/x-fits'],
		'fvt': ['video/vnd.fvt'],
		'fxm': ['video/x-javafx'],
		'fxp': ['application/vnd.adobe.fxp'],
		'fxpl': ['application/vnd.adobe.fxp'],
		'fzs': ['application/vnd.fuzzysheet'],
		'g2w': ['application/vnd.geoplan'],
		'g3': ['image/fax-g3', 'image/g3fax'],
		'g3w': ['application/vnd.geospace'],
		'gac': ['application/vnd.groove-account'],
		'gam': ['application/x-tads'],
		'gb': ['application/x-gameboy-rom'],
		'gba': ['application/x-gba-rom'],
		'gbc': ['application/x-gameboy-color-rom'],
		'gbr': ['application/rpki-ghostbusters', 'application/vnd.gerber', 'application/x-gerber', 'image/x-gimp-gbr'],
		'gbrjob': ['application/x-gerber-job'],
		'gca': ['application/x-gca-compressed'],
		'gcode': ['text/x.gcode'],
		'gcrd': ['text/directory', 'text/vcard', 'text/x-vcard'],
		'gd': ['application/x-gdscript'],
		'gdi': ['application/x-gd-rom-cue'],
		'gdl': ['model/vnd.gdl'],
		'gdoc': ['application/vnd.google-apps.document'],
		'gdshader': ['application/x-godot-shader'],
		'ged': ['application/x-gedcom', 'text/gedcom', 'text/vnd.familysearch.gedcom'],
		'gedcom': ['application/x-gedcom', 'text/gedcom', 'text/vnd.familysearch.gedcom'],
		'gem': ['application/x-gtar', 'application/x-tar'],
		'gen': ['application/x-genesis-rom'],
		'geo': ['application/vnd.dynageo'],
		'geo.json': ['application/geo+json', 'application/vnd.geo+json'],
		'geojson': ['application/geo+json', 'application/vnd.geo+json'],
		'gex': ['application/vnd.geometry-explorer'],
		'gf': ['application/x-tex-gf'],
		'gg': ['application/x-gamegear-rom'],
		'ggb': ['application/vnd.geogebra.file'],
		'ggs': ['application/vnd.geogebra.slides'],
		'ggt': ['application/vnd.geogebra.tool'],
		'ghf': ['application/vnd.groove-help'],
		'gif': ['image/gif'],
		'gih': ['image/x-gimp-gih'],
		'gim': ['application/vnd.groove-identity-message'],
		'glade': ['application/x-glade'],
		'glb': ['model/gltf-binary'],
		'gltf': ['model/gltf+json'],
		'gml': ['application/gml+xml'],
		'gmo': ['application/x-gettext-translation'],
		'gmx': ['application/vnd.gmx'],
		'gnc': ['application/x-gnucash'],
		'gnd': ['application/gnunet-directory'],
		'gnucash': ['application/x-gnucash'],
		'gnumeric': ['application/x-gnumeric'],
		'gnuplot': ['application/x-gnuplot'],
		'go': ['text/x-go'],
		'gp': ['application/x-gnuplot'],
		'gpg': ['application/pgp', 'application/pgp-encrypted', 'application/pgp-keys', 'application/pgp-signature'],
		'gph': ['application/vnd.flographit'],
		'gplt': ['application/x-gnuplot'],
		'gpx': ['application/gpx', 'application/gpx+xml', 'application/x-gpx', 'application/x-gpx+xml'],
		'gqf': ['application/vnd.grafeq'],
		'gqs': ['application/vnd.grafeq'],
		'gra': ['application/x-graphite'],
		'gradle': ['text/x-gradle'],
		'gram': ['application/srgs'],
		'gramps': ['application/x-gramps-xml'],
		'gre': ['application/vnd.geometry-explorer'],
		'groovy': ['text/x-groovy'],
		'grv': ['application/vnd.groove-injector'],
		'grxml': ['application/srgs+xml'],
		'gs': ['text/x-genie'],
		'gsf': ['application/x-font-ghostscript', 'application/x-font-type1'],
		'gsh': ['text/x-groovy'],
		'gsheet': ['application/vnd.google-apps.spreadsheet'],
		'gslides': ['application/vnd.google-apps.presentation'],
		'gsm': ['audio/x-gsm'],
		'gtar': ['application/x-gtar', 'application/x-tar'],
		'gtm': ['application/vnd.groove-tool-message'],
		'gtw': ['model/vnd.gtw'],
		'gv': ['text/vnd.graphviz'],
		'gvp': ['text/google-video-pointer', 'text/x-google-video-pointer'],
		'gvy': ['text/x-groovy'],
		'gx': ['text/x-gcode-gx'],
		'gxf': ['application/gxf'],
		'gxt': ['application/vnd.geonext'],
		'gy': ['text/x-groovy'],
		'gz': ['application/x-gzip', 'application/gzip'],
		'h': ['text/x-c', 'text/x-chdr'],
		'h++': ['text/x-c++hdr'],
		'h261': ['video/h261'],
		'h263': ['video/h263'],
		'h264': ['video/h264'],
		'h4': ['application/x-hdf'],
		'h5': ['application/x-hdf'],
		'hal': ['application/vnd.hal+xml'],
		'hbci': ['application/vnd.hbci'],
		'hbs': ['text/x-handlebars-template'],
		'hdd': ['application/x-virtualbox-hdd'],
		'hdf': ['application/x-hdf'],
		'hdf4': ['application/x-hdf'],
		'hdf5': ['application/x-hdf'],
		'hdp': ['image/jxr', 'image/vnd.ms-photo'],
		'hdv': ['video/mpeg'],
		'heic': ['image/heic', 'image/heic-sequence', 'image/heif', 'image/heif-sequence'],
		'heics': ['image/heic-sequence'],
		'heif': ['image/heic', 'image/heic-sequence', 'image/heif', 'image/heif-sequence'],
		'heifs': ['image/heif-sequence'],
		'hej2': ['image/hej2k'],
		'held': ['application/atsc-held+xml'],
		'hfe': ['application/x-hfe-file', 'application/x-hfe-floppy-image'],
		'hh': ['text/x-c', 'text/x-c++hdr'],
		'hif': ['image/heic', 'image/heic-sequence', 'image/heif', 'image/heif-sequence'],
		'hjson': ['application/hjson'],
		'hlp': ['application/winhlp', 'zz-application/zz-winassoc-hlp'],
		'hp': ['text/x-c++hdr'],
		'hpgl': ['application/vnd.hp-hpgl'],
		'hpid': ['application/vnd.hp-hpid'],
		'hpp': ['text/x-c++hdr'],
		'hps': ['application/vnd.hp-hps'],
		'hqx': ['application/stuffit', 'application/mac-binhex40'],
		'hs': ['text/x-haskell'],
		'hsj2': ['image/hsj2'],
		'hta': ['application/hta'],
		'htc': ['text/x-component'],
		'htke': ['application/vnd.kenameaapp'],
		'htm': ['text/html', 'application/xhtml+xml'],
		'html': ['text/html', 'application/xhtml+xml'],
		'hvd': ['application/vnd.yamaha.hv-dic'],
		'hvp': ['application/vnd.yamaha.hv-voice'],
		'hvs': ['application/vnd.yamaha.hv-script'],
		'hwp': ['application/vnd.haansoft-hwp', 'application/x-hwp'],
		'hwt': ['application/vnd.haansoft-hwt', 'application/x-hwt'],
		'hxx': ['text/x-c++hdr'],
		'i2g': ['application/vnd.intergeo'],
		'ica': ['application/x-ica'],
		'icalendar': ['application/ics', 'text/calendar', 'text/x-vcalendar'],
		'icb': ['application/tga', 'application/x-targa', 'application/x-tga', 'image/targa', 'image/tga', 'image/x-icb', 'image/x-targa', 'image/x-tga'],
		'icc': ['application/vnd.iccprofile'],
		'ice': ['x-conference/x-cooltalk'],
		'icm': ['application/vnd.iccprofile'],
		'icns': ['image/x-icns'],
		'ico': ['application/ico', 'image/ico', 'image/icon', 'image/vnd.microsoft.icon', 'image/x-ico', 'image/x-icon', 'text/ico'],
		'ics': ['application/ics', 'text/calendar', 'text/x-vcalendar'],
		'idl': ['text/x-idl'],
		'ief': ['image/ief'],
		'ifb': ['application/ics', 'text/calendar', 'text/x-vcalendar'],
		'iff': ['image/x-iff', 'image/x-ilbm'],
		'ifm': ['application/vnd.shana.informed.formdata'],
		'iges': ['model/iges'],
		'igl': ['application/vnd.igloader'],
		'igm': ['application/vnd.insors.igm'],
		'igs': ['model/iges'],
		'igx': ['application/vnd.micrografx.igx'],
		'iif': ['application/vnd.shana.informed.interchange'],
		'ilbm': ['image/x-iff', 'image/x-ilbm'],
		'ime': ['audio/imelody', 'audio/x-imelody', 'text/x-imelody'],
		'img': ['application/vnd.efi.img', 'application/x-raw-disk-image'],
		'img.xz': ['application/x-raw-disk-image-xz-compressed'],
		'imp': ['application/vnd.accpac.simply.imp'],
		'ims': ['application/vnd.ms-ims'],
		'imy': ['audio/imelody', 'audio/x-imelody', 'text/x-imelody'],
		'in': ['text/plain'],
		'ini': ['text/plain'],
		'ink': ['application/inkml+xml'],
		'inkml': ['application/inkml+xml'],
		'ins': ['application/x-tex', 'text/x-tex'],
		'install': ['application/x-install-instructions'],
		'iota': ['application/vnd.astraea-software.iota'],
		'ipfix': ['application/ipfix'],
		'ipk': ['application/vnd.shana.informed.package'],
		'ips': ['application/x-ips-patch'],
		'iptables': ['text/x-iptables'],
		'ipynb': ['application/x-ipynb+json'],
		'irm': ['application/vnd.ibm.rights-management'],
		'irp': ['application/vnd.irepository.package+xml'],
		'iso': ['application/vnd.efi.iso', 'application/x-cd-image', 'application/x-dreamcast-rom', 'application/x-gamecube-iso-image', 'application/x-gamecube-rom', 'application/x-iso9660-image', 'application/x-saturn-rom', 'application/x-sega-cd-rom', 'application/x-sega-pico-rom', 'application/x-wbfs', 'application/x-wia', 'application/x-wii-iso-image', 'application/x-wii-rom'],
		'iso9660': ['application/vnd.efi.iso', 'application/x-cd-image', 'application/x-iso9660-image'],
		'it': ['audio/x-it'],
		'it87': ['application/x-it87'],
		'itp': ['application/vnd.shana.informed.formtemplate'],
		'its': ['application/its+xml'],
		'ivp': ['application/vnd.immervision-ivp'],
		'ivu': ['application/vnd.immervision-ivu'],
		'j2c': ['image/x-jp2-codestream'],
		'j2k': ['image/x-jp2-codestream'],
		'jad': ['text/vnd.sun.j2me.app-descriptor'],
		'jade': ['text/jade'],
		'jam': ['application/vnd.jam'],
		'jar': ['application/x-java-archive', 'application/java-archive', 'application/x-jar'],
		'jardiff': ['application/x-java-archive-diff'],
		'java': ['text/x-java', 'text/x-java-source'],
		'jceks': ['application/x-java-jce-keystore'],
		'jfif': ['image/jpeg', 'image/pjpeg'],
		'jhc': ['image/jphc'],
		'jisp': ['application/vnd.jisp'],
		'jks': ['application/x-java-keystore'],
		'jl': ['text/julia'],
		'jls': ['image/jls'],
		'jlt': ['application/vnd.hp-jlyt'],
		'jng': ['image/x-jng'],
		'jnlp': ['application/x-java-jnlp-file'],
		'joda': ['application/vnd.joost.joda-archive'],
		'jp2': ['image/jp2', 'image/jpeg2000', 'image/jpeg2000-image', 'image/x-jpeg2000-image'],
		'jpc': ['image/x-jp2-codestream'],
		'jpe': ['image/jpeg', 'image/pjpeg'],
		'jpeg': ['image/jpeg', 'image/pjpeg'],
		'jpf': ['image/jpx'],
		'jpg': ['image/jpeg', 'image/pjpeg'],
		'jpg2': ['image/jp2', 'image/jpeg2000', 'image/jpeg2000-image', 'image/x-jpeg2000-image'],
		'jpgm': ['image/jpm', 'video/jpm'],
		'jpgv': ['video/jpeg'],
		'jph': ['image/jph'],
		'jpm': ['image/jpm', 'video/jpm'],
		'jpr': ['application/x-jbuilder-project'],
		'jpx': ['application/x-jbuilder-project', 'image/jpx'],
		'jrd': ['application/jrd+json'],
		'js': ['text/javascript', 'application/javascript', 'application/x-javascript', 'text/jscript'],
		'jse': ['text/jscript.encode'],
		'jsm': ['application/javascript', 'application/x-javascript', 'text/javascript', 'text/jscript'],
		'json': ['application/json', 'application/schema+json'],
		'json-patch': ['application/json-patch+json'],
		'json5': ['application/json5'],
		'jsonld': ['application/ld+json'],
		'jsonml': ['application/jsonml+json'],
		'jsx': ['text/jsx'],
		'jt': ['model/jt'],
		'jxl': ['image/jxl'],
		'jxr': ['image/jxr', 'image/vnd.ms-photo'],
		'jxra': ['image/jxra'],
		'jxrs': ['image/jxrs'],
		'jxs': ['image/jxs'],
		'jxsc': ['image/jxsc'],
		'jxsi': ['image/jxsi'],
		'jxss': ['image/jxss'],
		'k25': ['image/x-kodak-k25'],
		'k7': ['application/x-thomson-cassette'],
		'kar': ['audio/midi', 'audio/x-midi'],
		'karbon': ['application/vnd.kde.karbon', 'application/x-karbon'],
		'kcf': ['image/x-kiss-cel'],
		'kdbx': ['application/x-keepass2'],
		'kdc': ['image/x-kodak-kdc'],
		'kdelnk': ['application/x-desktop', 'application/x-gnome-app-info'],
		'kexi': ['application/x-kexiproject-sqlite', 'application/x-kexiproject-sqlite2', 'application/x-kexiproject-sqlite3', 'application/x-vnd.kde.kexi'],
		'kexic': ['application/x-kexi-connectiondata'],
		'kexis': ['application/x-kexiproject-shortcut'],
		'key': ['application/vnd.apple.keynote', 'application/pgp-keys', 'application/x-iwork-keynote-sffkey'],
		'keynote': ['application/vnd.apple.keynote'],
		'kfo': ['application/vnd.kde.kformula', 'application/x-kformula'],
		'kfx': ['application/vnd.amazon.mobi8-ebook', 'application/x-mobi8-ebook'],
		'kia': ['application/vnd.kidspiration'],
		'kil': ['application/x-killustrator'],
		'kino': ['application/smil', 'application/smil+xml'],
		'kml': ['application/vnd.google-earth.kml+xml'],
		'kmz': ['application/vnd.google-earth.kmz'],
		'kne': ['application/vnd.kinar'],
		'knp': ['application/vnd.kinar'],
		'kon': ['application/vnd.kde.kontour', 'application/x-kontour'],
		'kpm': ['application/x-kpovmodeler'],
		'kpr': ['application/vnd.kde.kpresenter', 'application/x-kpresenter'],
		'kpt': ['application/vnd.kde.kpresenter', 'application/x-kpresenter'],
		'kpxx': ['application/vnd.ds-keypoint'],
		'kra': ['application/x-krita'],
		'krz': ['application/x-krita'],
		'ks': ['application/x-java-keystore'],
		'ksp': ['application/vnd.kde.kspread', 'application/x-kspread'],
		'ksy': ['text/x-kaitai-struct'],
		'kt': ['text/x-kotlin'],
		'ktr': ['application/vnd.kahootz'],
		'ktx': ['image/ktx'],
		'ktx2': ['image/ktx2'],
		'ktz': ['application/vnd.kahootz'],
		'kud': ['application/x-kugar'],
		'kwd': ['application/vnd.kde.kword', 'application/x-kword'],
		'kwt': ['application/vnd.kde.kword', 'application/x-kword'],
		'la': ['application/x-shared-library-la'],
		'lasxml': ['application/vnd.las.las+xml'],
		'latex': ['application/x-latex', 'application/x-tex', 'text/x-tex'],
		'lbd': ['application/vnd.llamagraphics.life-balance.desktop'],
		'lbe': ['application/vnd.llamagraphics.life-balance.exchange+xml'],
		'lbm': ['image/x-iff', 'image/x-ilbm'],
		'ldif': ['text/x-ldif'],
		'les': ['application/vnd.hhe.lesson-player'],
		'less': ['text/less'],
		'lgr': ['application/lgr+xml'],
		'lha': ['application/x-lha', 'application/x-lzh-compressed'],
		'lhs': ['text/x-literate-haskell'],
		'lhz': ['application/x-lhz'],
		'lib': ['application/vnd.microsoft.portable-executable', 'application/x-archive'],
		'link66': ['application/vnd.route66.link66+xml'],
		'lisp': ['text/x-common-lisp'],
		'list': ['text/plain'],
		'list3820': ['application/vnd.ibm.modcap'],
		'listafp': ['application/vnd.ibm.modcap'],
		'litcoffee': ['text/coffeescript'],
		'lmdb': ['application/x-lmdb'],
		'lnk': ['application/x-ms-shortcut', 'application/x-win-lnk'],
		'lnx': ['application/x-atari-lynx-rom'],
		'loas': ['audio/usac'],
		'log': ['text/plain', 'text/x-log'],
		'lostxml': ['application/lost+xml'],
		'lrf': ['application/x-sony-bbeb'],
		'lrm': ['application/vnd.ms-lrm'],
		'lrv': ['video/mp4', 'video/mp4v-es', 'video/x-m4v'],
		'lrz': ['application/x-lrzip'],
		'ltf': ['application/vnd.frogans.ltf'],
		'ltx': ['application/x-tex', 'text/x-tex'],
		'lua': ['text/x-lua'],
		'luac': ['application/x-lua-bytecode'],
		'lvp': ['audio/vnd.lucent.voice'],
		'lwo': ['image/x-lwo'],
		'lwob': ['image/x-lwo'],
		'lwp': ['application/vnd.lotus-wordpro'],
		'lws': ['image/x-lws'],
		'ly': ['text/x-lilypond'],
		'lyx': ['application/x-lyx', 'text/x-lyx'],
		'lz': ['application/x-lzip'],
		'lz4': ['application/x-lz4'],
		'lzh': ['application/x-lha', 'application/x-lzh-compressed'],
		'lzma': ['application/x-lzma'],
		'lzo': ['application/x-lzop'],
		'm': ['text/x-matlab', 'text/x-objcsrc', 'text/x-octave'],
		'm13': ['application/x-msmediaview'],
		'm14': ['application/x-msmediaview'],
		'm15': ['audio/x-mod'],
		'm1u': ['video/vnd.mpegurl', 'video/x-mpegurl'],
		'm1v': ['video/mpeg'],
		'm21': ['application/mp21'],
		'm2a': ['audio/mpeg'],
		'm2t': ['video/mp2t'],
		'm2ts': ['video/mp2t'],
		'm2v': ['video/mpeg'],
		'm3a': ['audio/mpeg'],
		'm3u': ['audio/x-mpegurl', 'application/m3u', 'application/vnd.apple.mpegurl', 'audio/m3u', 'audio/mpegurl', 'audio/x-m3u', 'audio/x-mp3-playlist'],
		'm3u8': ['application/m3u', 'application/vnd.apple.mpegurl', 'audio/m3u', 'audio/mpegurl', 'audio/x-m3u', 'audio/x-mp3-playlist', 'audio/x-mpegurl'],
		'm4': ['application/x-m4'],
		'm4a': ['audio/mp4', 'audio/m4a', 'audio/x-m4a'],
		'm4b': ['audio/x-m4b'],
		'm4p': ['application/mp4'],
		'm4r': ['audio/x-m4r'],
		'm4s': ['video/iso.segment'],
		'm4u': ['video/vnd.mpegurl', 'video/x-mpegurl'],
		'm4v': ['video/mp4', 'video/mp4v-es', 'video/x-m4v'],
		'm7': ['application/x-thomson-cartridge-memo7'],
		'ma': ['application/mathematica'],
		'mab': ['application/x-markaby'],
		'mads': ['application/mads+xml'],
		'maei': ['application/mmt-aei+xml'],
		'mag': ['application/vnd.ecowin.chart'],
		'mak': ['text/x-makefile'],
		'maker': ['application/vnd.framemaker'],
		'man': ['application/x-troff-man', 'text/troff'],
		'manifest': ['text/cache-manifest'],
		'map': ['application/json'],
		'markdown': ['text/markdown', 'text/x-markdown'],
		'mathml': ['application/mathml+xml'],
		'mb': ['application/mathematica'],
		'mbk': ['application/vnd.mobius.mbk'],
		'mbox': ['application/mbox'],
		'mc1': ['application/vnd.medcalcdata'],
		'mc2': ['text/vnd.senx.warpscript'],
		'mcd': ['application/vnd.mcd'],
		'mcurl': ['text/vnd.curl.mcurl'],
		'md': ['text/markdown', 'text/x-markdown', 'application/x-genesis-rom'],
		'mdb': ['application/x-msaccess', 'application/mdb', 'application/msaccess', 'application/vnd.ms-access', 'application/vnd.msaccess', 'application/x-lmdb', 'application/x-mdb', 'zz-application/zz-winassoc-mdb'],
		'mdi': ['image/vnd.ms-modi'],
		'mdx': ['application/x-genesis-32x-rom', 'text/mdx'],
		'me': ['text/troff', 'text/x-troff-me'],
		'med': ['audio/x-mod'],
		'mesh': ['model/mesh'],
		'meta4': ['application/metalink4+xml'],
		'metalink': ['application/metalink+xml'],
		'mets': ['application/mets+xml'],
		'mfm': ['application/vnd.mfmp'],
		'mft': ['application/rpki-manifest'],
		'mgp': ['application/vnd.osgeo.mapguide.package', 'application/x-magicpoint'],
		'mgz': ['application/vnd.proteus.magazine'],
		'mht': ['application/x-mimearchive'],
		'mhtml': ['application/x-mimearchive'],
		'mid': ['audio/midi', 'audio/x-midi'],
		'midi': ['audio/midi', 'audio/x-midi'],
		'mie': ['application/x-mie'],
		'mif': ['application/vnd.mif', 'application/x-mif'],
		'mime': ['message/rfc822'],
		'minipsf': ['audio/x-minipsf'],
		'mj2': ['video/mj2'],
		'mjp2': ['video/mj2'],
		'mjpeg': ['video/x-mjpeg'],
		'mjpg': ['video/x-mjpeg'],
		'mjs': ['application/javascript', 'application/x-javascript', 'text/javascript', 'text/jscript'],
		'mk': ['text/x-makefile'],
		'mk3d': ['video/x-matroska', 'video/x-matroska-3d'],
		'mka': ['audio/x-matroska'],
		'mkd': ['text/markdown', 'text/x-markdown'],
		'mks': ['video/x-matroska'],
		'mkv': ['video/x-matroska'],
		'ml': ['text/x-ocaml'],
		'mli': ['text/x-ocaml'],
		'mlp': ['application/vnd.dolby.mlp'],
		'mm': ['text/x-objc++src', 'text/x-troff-mm'],
		'mmd': ['application/vnd.chipnuts.karaoke-mmd'],
		'mmf': ['application/vnd.smaf', 'application/x-smaf'],
		'mml': ['application/mathml+xml', 'text/mathml'],
		'mmr': ['image/vnd.fujixerox.edmics-mmr'],
		'mng': ['video/x-mng'],
		'mny': ['application/x-msmoney'],
		'mo': ['application/x-gettext-translation', 'text/x-modelica'],
		'mo3': ['audio/x-mo3'],
		'mobi': ['application/x-mobipocket-ebook'],
		'moc': ['text/x-moc'],
		'mod': ['application/x-object', 'audio/x-mod'],
		'mods': ['application/mods+xml'],
		'mof': ['text/x-mof'],
		'moov': ['video/quicktime'],
		'mount': ['text/x-systemd-unit'],
		'mov': ['video/quicktime'],
		'movie': ['video/x-sgi-movie'],
		'mp+': ['audio/x-musepack'],
		'mp2': ['audio/mp2', 'audio/mpeg', 'audio/x-mp2', 'video/mpeg', 'video/mpeg-system', 'video/x-mpeg', 'video/x-mpeg-system', 'video/x-mpeg2'],
		'mp21': ['application/mp21'],
		'mp2a': ['audio/mpeg'],
		'mp3': ['audio/mpeg', 'audio/mp3', 'audio/x-mp3', 'audio/x-mpeg', 'audio/x-mpg'],
		'mp4': ['video/mp4', 'application/mp4', 'video/mp4v-es', 'video/x-m4v'],
		'mp4a': ['audio/mp4'],
		'mp4s': ['application/mp4'],
		'mp4v': ['video/mp4'],
		'mpc': ['application/vnd.mophun.certificate', 'audio/x-musepack'],
		'mpd': ['application/dash+xml'],
		'mpe': ['video/mpeg', 'video/mpeg-system', 'video/x-mpeg', 'video/x-mpeg-system', 'video/x-mpeg2'],
		'mpeg': ['video/mpeg', 'video/mpeg-system', 'video/x-mpeg', 'video/x-mpeg-system', 'video/x-mpeg2'],
		'mpf': ['application/media-policy-dataset+xml'],
		'mpg': ['video/mpeg', 'video/mpeg-system', 'video/x-mpeg', 'video/x-mpeg-system', 'video/x-mpeg2'],
		'mpg4': ['video/mpg4', 'application/mp4', 'video/mp4'],
		'mpga': ['audio/mp3', 'audio/mpeg', 'audio/x-mp3', 'audio/x-mpeg', 'audio/x-mpg'],
		'mpkg': ['application/vnd.apple.installer+xml'],
		'mpl': ['text/x-mpl2', 'video/mp2t'],
		'mpls': ['video/mp2t'],
		'mpm': ['application/vnd.blueice.multipass'],
		'mpn': ['application/vnd.mophun.application'],
		'mpp': ['application/dash-patch+xml', 'application/vnd.ms-project', 'audio/x-musepack'],
		'mpt': ['application/vnd.ms-project'],
		'mpy': ['application/vnd.ibm.minipay'],
		'mqy': ['application/vnd.mobius.mqy'],
		'mrc': ['application/marc'],
		'mrcx': ['application/marcxml+xml'],
		'mrl': ['text/x-mrml'],
		'mrml': ['text/x-mrml'],
		'mrpack': ['application/x-modrinth-modpack+zip'],
		'mrw': ['image/x-minolta-mrw'],
		'ms': ['text/troff', 'text/x-troff-ms'],
		'mscml': ['application/mediaservercontrol+xml'],
		'mseed': ['application/vnd.fdsn.mseed'],
		'mseq': ['application/vnd.mseq'],
		'msf': ['application/vnd.epson.msf'],
		'msg': ['application/vnd.ms-outlook'],
		'msh': ['model/mesh'],
		'msi': ['application/x-msdownload', 'application/x-msi'],
		'msix': ['application/msix'],
		'msixbundle': ['application/msixbundle'],
		'msl': ['application/vnd.mobius.msl'],
		'msod': ['image/x-msod'],
		'msp': ['application/microsoftpatch'],
		'msty': ['application/vnd.muvee.style'],
		'msu': ['application/microsoftupdate'],
		'msx': ['application/x-msx-rom'],
		'mtl': ['model/mtl'],
		'mtm': ['audio/x-mod'],
		'mts': ['model/vnd.mts', 'video/mp2t'],
		'mup': ['text/x-mup'],
		'mus': ['application/vnd.musician'],
		'musd': ['application/mmt-usd+xml'],
		'musicxml': ['application/vnd.recordare.musicxml+xml'],
		'mvb': ['application/x-msmediaview'],
		'mvt': ['application/vnd.mapbox-vector-tile'],
		'mwf': ['application/vnd.mfer'],
		'mxf': ['application/mxf'],
		'mxl': ['application/vnd.recordare.musicxml'],
		'mxmf': ['audio/mobile-xmf', 'audio/vnd.nokia.mobile-xmf'],
		'mxml': ['application/xv+xml'],
		'mxs': ['application/vnd.triscape.mxs'],
		'mxu': ['video/vnd.mpegurl', 'video/x-mpegurl'],
		'n-gage': ['application/vnd.nokia.n-gage.symbian.install'],
		'n3': ['text/n3'],
		'n64': ['application/x-n64-rom'],
		'nb': ['application/mathematica', 'application/x-mathematica'],
		'nbp': ['application/vnd.wolfram.player'],
		'nc': ['application/x-netcdf'],
		'ncx': ['application/x-dtbncx+xml'],
		'nds': ['application/x-nintendo-ds-rom'],
		'nef': ['image/x-nikon-nef'],
		'nes': ['application/x-nes-rom'],
		'nez': ['application/x-nes-rom'],
		'nfo': ['text/x-nfo'],
		'ngc': ['application/x-neo-geo-pocket-color-rom'],
		'ngdat': ['application/vnd.nokia.n-gage.data'],
		'ngp': ['application/x-neo-geo-pocket-rom'],
		'nim': ['text/x-nim'],
		'nimble': ['text/x-nimscript'],
		'nims': ['text/x-nimscript'],
		'nitf': ['application/vnd.nitf'],
		'nix': ['text/x-nix'],
		'nlu': ['application/vnd.neurolanguage.nlu'],
		'nml': ['application/vnd.enliven'],
		'nnd': ['application/vnd.noblenet-directory'],
		'nns': ['application/vnd.noblenet-sealer'],
		'nnw': ['application/vnd.noblenet-web'],
		'not': ['text/x-mup'],
		'npx': ['image/vnd.net-fpx'],
		'nq': ['application/n-quads'],
		'nrw': ['image/x-nikon-nrw'],
		'nsc': ['application/x-conference', 'application/x-netshow-channel'],
		'nsf': ['application/vnd.lotus-notes'],
		'nsv': ['video/x-nsv'],
		'nt': ['application/n-triples'],
		'ntar': ['application/x-pcapng'],
		'ntf': ['application/vnd.nitf'],
		'nu': ['application/x-nuscript', 'text/x-nu'],
		'numbers': ['application/vnd.apple.numbers', 'application/x-iwork-numbers-sffnumbers'],
		'nzb': ['application/x-nzb'],
		'o': ['application/x-object'],
		'oa2': ['application/vnd.fujitsu.oasys2'],
		'oa3': ['application/vnd.fujitsu.oasys3'],
		'oas': ['application/vnd.fujitsu.oasys'],
		'obd': ['application/x-msbinder'],
		'obgx': ['application/vnd.openblox.game+xml'],
		'obj': ['application/prs.wavefront-obj', 'application/x-tgif', 'model/obj'],
		'ocl': ['text/x-ocl'],
		'ocx': ['application/vnd.microsoft.portable-executable'],
		'oda': ['application/oda'],
		'odb': ['application/vnd.oasis.opendocument.base', 'application/vnd.oasis.opendocument.database', 'application/vnd.sun.xml.base'],
		'odc': ['application/vnd.oasis.opendocument.chart'],
		'odf': ['application/vnd.oasis.opendocument.formula'],
		'odft': ['application/vnd.oasis.opendocument.formula-template'],
		'odg': ['application/vnd.oasis.opendocument.graphics'],
		'odi': ['application/vnd.oasis.opendocument.image'],
		'odm': ['application/vnd.oasis.opendocument.text-master'],
		'odp': ['application/vnd.oasis.opendocument.presentation'],
		'ods': ['application/vnd.oasis.opendocument.spreadsheet'],
		'odt': ['application/vnd.oasis.opendocument.text'],
		'oga': ['audio/ogg', 'audio/vorbis', 'audio/x-flac+ogg', 'audio/x-ogg', 'audio/x-oggflac', 'audio/x-speex+ogg', 'audio/x-vorbis', 'audio/x-vorbis+ogg'],
		'ogex': ['model/vnd.opengex'],
		'ogg': ['audio/ogg', 'audio/vorbis', 'audio/x-flac+ogg', 'audio/x-ogg', 'audio/x-oggflac', 'audio/x-speex+ogg', 'audio/x-vorbis', 'audio/x-vorbis+ogg', 'video/ogg', 'video/x-ogg', 'video/x-theora', 'video/x-theora+ogg'],
		'ogm': ['video/x-ogm', 'video/x-ogm+ogg'],
		'ogv': ['video/ogg', 'video/x-ogg'],
		'ogx': ['application/ogg', 'application/x-ogg'],
		'old': ['application/x-trash'],
		'oleo': ['application/x-oleo'],
		'omdoc': ['application/omdoc+xml'],
		'onepkg': ['application/onenote'],
		'onetmp': ['application/onenote'],
		'onetoc': ['application/onenote'],
		'onetoc2': ['application/onenote'],
		'ooc': ['text/x-ooc'],
		'openvpn': ['application/x-openvpn-profile'],
		'opf': ['application/oebps-package+xml'],
		'opml': ['text/x-opml', 'text/x-opml+xml'],
		'oprc': ['application/vnd.palm', 'application/x-palm-database'],
		'opus': ['audio/ogg', 'audio/x-ogg', 'audio/x-opus+ogg'],
		'ora': ['image/openraster'],
		'orf': ['image/x-olympus-orf'],
		'org': ['application/vnd.lotus-organizer', 'text/org', 'text/x-org'],
		'osf': ['application/vnd.yamaha.openscoreformat'],
		'osfpvg': ['application/vnd.yamaha.openscoreformat.osfpvg+xml'],
		'osm': ['application/vnd.openstreetmap.data+xml'],
		'otc': ['application/vnd.oasis.opendocument.chart-template'],
		'otf': ['application/vnd.oasis.opendocument.formula-template', 'application/x-font-otf', 'font/otf'],
		'otg': ['application/vnd.oasis.opendocument.graphics-template'],
		'oth': ['application/vnd.oasis.opendocument.text-web'],
		'oti': ['application/vnd.oasis.opendocument.image-template'],
		'otm': ['application/vnd.oasis.opendocument.text-master-template'],
		'otp': ['application/vnd.oasis.opendocument.presentation-template'],
		'ots': ['application/vnd.oasis.opendocument.spreadsheet-template'],
		'ott': ['application/vnd.oasis.opendocument.text-template'],
		'ova': ['application/ovf', 'application/x-virtualbox-ova'],
		'ovf': ['application/x-virtualbox-ovf'],
		'ovpn': ['application/x-openvpn-profile'],
		'owl': ['application/rdf+xml', 'text/rdf'],
		'owx': ['application/owl+xml'],
		'oxps': ['application/oxps'],
		'oxt': ['application/vnd.openofficeorg.extension'],
		'p': ['text/x-pascal'],
		'p10': ['application/pkcs10'],
		'p12': ['application/pkcs12', 'application/x-pkcs12'],
		'p65': ['application/x-pagemaker'],
		'p7b': ['application/x-pkcs7-certificates'],
		'p7c': ['application/pkcs7-mime'],
		'p7m': ['application/pkcs7-mime'],
		'p7r': ['application/x-pkcs7-certreqresp'],
		'p7s': ['application/pkcs7-signature'],
		'p8': ['application/pkcs8'],
		'p8e': ['application/pkcs8-encrypted'],
		'pac': ['application/x-ns-proxy-autoconfig'],
		'pack': ['application/x-java-pack200'],
		'pages': ['application/vnd.apple.pages', 'application/x-iwork-pages-sffpages'],
		'pak': ['application/x-pak'],
		'par2': ['application/x-par2'],
		'parquet': ['application/vnd.apache.parquet', 'application/x-parquet'],
		'part': ['application/x-partial-download'],
		'pas': ['text/x-pascal'],
		'pat': ['image/x-gimp-pat'],
		'patch': ['text/x-diff', 'text/x-patch'],
		'path': ['text/x-systemd-unit'],
		'paw': ['application/vnd.pawaafile'],
		'pbd': ['application/vnd.powerbuilder6'],
		'pbm': ['image/x-portable-bitmap'],
		'pcap': ['application/pcap', 'application/vnd.tcpdump.pcap', 'application/x-pcap'],
		'pcapng': ['application/x-pcapng'],
		'pcd': ['image/x-photo-cd'],
		'pce': ['application/x-pc-engine-rom'],
		'pcf': ['application/x-cisco-vpn-settings', 'application/x-font-pcf'],
		'pcf.Z': ['application/x-font-pcf'],
		'pcf.gz': ['application/x-font-pcf'],
		'pcl': ['application/vnd.hp-pcl'],
		'pclxl': ['application/vnd.hp-pclxl'],
		'pct': ['image/x-pict'],
		'pcurl': ['application/vnd.curl.pcurl'],
		'pcx': ['image/vnd.zbrush.pcx', 'image/x-pcx'],
		'pdb': ['application/vnd.palm', 'application/x-aportisdoc', 'application/x-ms-pdb', 'application/x-palm-database', 'application/x-pilot', 'chemical/x-pdb'],
		'pdc': ['application/x-aportisdoc'],
		'pde': ['text/x-processing'],
		'pdf': ['application/pdf', 'application/acrobat', 'application/nappdf', 'application/x-pdf', 'image/pdf'],
		'pdf.bz2': ['application/x-bzpdf'],
		'pdf.gz': ['application/x-gzpdf'],
		'pdf.lz': ['application/x-lzpdf'],
		'pdf.xz': ['application/x-xzpdf'],
		'pef': ['image/x-pentax-pef'],
		'pem': ['application/x-x509-ca-cert'],
		'perl': ['application/x-perl', 'text/x-perl'],
		'pfa': ['application/x-font-type1'],
		'pfb': ['application/x-font-type1'],
		'pfm': ['application/x-font-type1', 'image/x-pfm'],
		'pfr': ['application/font-tdpfr', 'application/vnd.truedoc'],
		'pfx': ['application/pkcs12', 'application/x-pkcs12'],
		'pgm': ['image/x-portable-graymap'],
		'pgn': ['application/vnd.chess-pgn', 'application/x-chess-pgn'],
		'pgp': ['application/pgp', 'application/pgp-encrypted', 'application/pgp-keys', 'application/pgp-signature'],
		'php': ['application/x-php', 'application/x-httpd-php'],
		'php3': ['application/x-php'],
		'php4': ['application/x-php'],
		'php5': ['application/x-php'],
		'phps': ['application/x-php'],
		'pic': ['image/x-pict'],
		'pict': ['image/x-pict'],
		'pict1': ['image/x-pict'],
		'pict2': ['image/x-pict'],
		'pk': ['application/x-tex-pk'],
		'pkg': ['application/x-xar'],
		'pki': ['application/pkixcmp'],
		'pkipath': ['application/pkix-pkipath'],
		'pkpass': ['application/vnd.apple.pkpass'],
		'pkr': ['application/pgp-keys'],
		'pl': ['application/x-perl', 'text/x-perl'],
		'pla': ['audio/x-iriver-pla'],
		'plb': ['application/vnd.3gpp.pic-bw-large'],
		'plc': ['application/vnd.mobius.plc'],
		'plf': ['application/vnd.pocketlearn'],
		'pln': ['application/x-planperfect'],
		'pls': ['application/pls', 'application/pls+xml', 'audio/scpls', 'audio/x-scpls'],
		'pm': ['application/x-pagemaker', 'application/x-perl', 'text/x-perl'],
		'pm6': ['application/x-pagemaker'],
		'pmd': ['application/x-pagemaker'],
		'pml': ['application/vnd.ctc-posml'],
		'png': ['image/png', 'image/apng', 'image/vnd.mozilla.apng'],
		'pnm': ['image/x-portable-anymap'],
		'pntg': ['image/x-macpaint'],
		'po': ['application/x-gettext', 'text/x-gettext-translation', 'text/x-po'],
		'pod': ['application/x-perl', 'text/x-perl'],
		'por': ['application/x-spss-por'],
		'portpkg': ['application/vnd.macports.portpkg'],
		'pot': ['application/mspowerpoint', 'application/powerpoint', 'application/vnd.ms-powerpoint', 'application/x-mspowerpoint', 'text/x-gettext-translation-template', 'text/x-pot'],
		'potm': ['application/vnd.ms-powerpoint.template.macroenabled.12'],
		'potx': ['application/vnd.openxmlformats-officedocument.presentationml.template'],
		'ppam': ['application/vnd.ms-powerpoint.addin.macroenabled.12'],
		'ppd': ['application/vnd.cups-ppd'],
		'ppm': ['image/x-portable-pixmap'],
		'pps': ['application/mspowerpoint', 'application/powerpoint', 'application/vnd.ms-powerpoint', 'application/x-mspowerpoint'],
		'ppsm': ['application/vnd.ms-powerpoint.slideshow.macroenabled.12'],
		'ppsx': ['application/vnd.openxmlformats-officedocument.presentationml.slideshow'],
		'ppt': ['application/vnd.ms-powerpoint', 'application/mspowerpoint', 'application/powerpoint', 'application/x-mspowerpoint'],
		'pptm': ['application/vnd.ms-powerpoint.presentation.macroenabled.12'],
		'pptx': ['application/vnd.openxmlformats-officedocument.presentationml.presentation'],
		'ppz': ['application/mspowerpoint', 'application/powerpoint', 'application/vnd.ms-powerpoint', 'application/x-mspowerpoint'],
		'pqa': ['application/vnd.palm', 'application/x-palm-database'],
		'prc': ['application/vnd.palm', 'application/x-mobipocket-ebook', 'application/x-palm-database', 'application/x-pilot', 'model/prc'],
		'pre': ['application/vnd.lotus-freelance'],
		'prf': ['application/pics-rules'],
		'provx': ['application/provenance+xml'],
		'ps': ['application/postscript'],
		'ps.bz2': ['application/x-bzpostscript'],
		'ps.gz': ['application/x-gzpostscript'],
		'ps1': ['application/x-powershell'],
		'psb': ['application/vnd.3gpp.pic-bw-small'],
		'psd': ['application/photoshop', 'application/x-photoshop', 'image/photoshop', 'image/psd', 'image/vnd.adobe.photoshop', 'image/x-photoshop', 'image/x-psd'],
		'psf': ['application/x-font-linux-psf', 'audio/x-psf'],
		'psf.gz': ['application/x-gz-font-linux-psf'],
		'psflib': ['audio/x-psflib'],
		'psid': ['audio/prs.sid'],
		'pskcxml': ['application/pskc+xml'],
		'psw': ['application/x-pocket-word'],
		'pti': ['image/prs.pti'],
		'ptid': ['application/vnd.pvi.ptid1'],
		'pub': ['application/vnd.ms-publisher', 'application/x-mspublisher'],
		'pvb': ['application/vnd.3gpp.pic-bw-var'],
		'pw': ['application/x-pw'],
		'pwn': ['application/vnd.3m.post-it-notes'],
		'pxd': ['text/x-cython'],
		'pxi': ['text/x-cython'],
		'pxr': ['image/x-pxr'],
		'py': ['text/x-python', 'text/x-python2', 'text/x-python3'],
		'py2': ['text/x-python2'],
		'py3': ['text/x-python3'],
		'pya': ['audio/vnd.ms-playready.media.pya'],
		'pyc': ['application/x-python-bytecode'],
		'pyi': ['text/x-python3'],
		'pyo': ['application/x-python-bytecode', 'model/vnd.pytha.pyox'],
		'pyox': ['model/vnd.pytha.pyox'],
		'pys': ['application/x-pyspread-bz-spreadsheet'],
		'pysu': ['application/x-pyspread-spreadsheet'],
		'pyv': ['video/vnd.ms-playready.media.pyv'],
		'pyx': ['text/x-cython'],
		'qam': ['application/vnd.epson.quickanime'],
		'qbo': ['application/vnd.intu.qbo'],
		'qbrew': ['application/x-qbrew'],
		'qcow': ['application/x-qemu-disk'],
		'qcow2': ['application/x-qemu-disk'],
		'qd': ['application/x-fd-file', 'application/x-raw-floppy-disk-image'],
		'qed': ['application/x-qed-disk'],
		'qfx': ['application/vnd.intu.qfx'],
		'qif': ['application/x-qw', 'image/x-quicktime'],
		'qml': ['text/x-qml'],
		'qmlproject': ['text/x-qml'],
		'qmltypes': ['text/x-qml'],
		'qoi': ['image/qoi'],
		'qp': ['application/x-qpress'],
		'qps': ['application/vnd.publishare-delta-tree'],
		'qpw': ['application/x-quattropro'],
		'qs': ['application/sparql-query'],
		'qt': ['video/quicktime'],
		'qti': ['application/x-qtiplot'],
		'qti.gz': ['application/x-qtiplot'],
		'qtif': ['image/x-quicktime'],
		'qtl': ['application/x-quicktime-media-link', 'application/x-quicktimeplayer'],
		'qtvr': ['video/quicktime'],
		'qwd': ['application/vnd.quark.quarkxpress'],
		'qwt': ['application/vnd.quark.quarkxpress'],
		'qxb': ['application/vnd.quark.quarkxpress'],
		'qxd': ['application/vnd.quark.quarkxpress'],
		'qxl': ['application/vnd.quark.quarkxpress'],
		'qxp': ['application/vnd.quark.quarkxpress'],
		'qxt': ['application/vnd.quark.quarkxpress'],
		'ra': ['audio/vnd.m-realaudio', 'audio/vnd.rn-realaudio', 'audio/x-pn-realaudio', 'audio/x-realaudio'],
		'raf': ['image/x-fuji-raf'],
		'ram': ['application/ram', 'audio/x-pn-realaudio'],
		'raml': ['application/raml+yaml'],
		'rapd': ['application/route-apd+xml'],
		'rar': ['application/x-rar-compressed', 'application/vnd.rar', 'application/x-rar'],
		'ras': ['image/x-cmu-raster'],
		'raw': ['image/x-panasonic-raw', 'image/x-panasonic-rw'],
		'raw-disk-image': ['application/vnd.efi.img', 'application/x-raw-disk-image'],
		'raw-disk-image.xz': ['application/x-raw-disk-image-xz-compressed'],
		'rax': ['audio/vnd.m-realaudio', 'audio/vnd.rn-realaudio', 'audio/x-pn-realaudio'],
		'rb': ['application/x-ruby'],
		'rcprofile': ['application/vnd.ipunplugged.rcprofile'],
		'rdf': ['application/rdf+xml', 'text/rdf'],
		'rdfs': ['application/rdf+xml', 'text/rdf'],
		'rdz': ['application/vnd.data-vision.rdz'],
		'reg': ['text/x-ms-regedit'],
		'rej': ['application/x-reject', 'text/x-reject'],
		'relo': ['application/p2p-overlay+xml'],
		'rep': ['application/vnd.businessobjects'],
		'res': ['application/x-dtbresource+xml', 'application/x-godot-resource'],
		'rgb': ['image/x-rgb'],
		'rif': ['application/reginfo+xml'],
		'rip': ['audio/vnd.rip'],
		'ris': ['application/x-research-info-systems'],
		'rl': ['application/resource-lists+xml'],
		'rlc': ['image/vnd.fujixerox.edmics-rlc'],
		'rld': ['application/resource-lists-diff+xml'],
		'rle': ['image/rle'],
		'rm': ['application/vnd.rn-realmedia', 'application/vnd.rn-realmedia-vbr'],
		'rmi': ['audio/midi'],
		'rmj': ['application/vnd.rn-realmedia', 'application/vnd.rn-realmedia-vbr'],
		'rmm': ['application/vnd.rn-realmedia', 'application/vnd.rn-realmedia-vbr'],
		'rmp': ['audio/x-pn-realaudio-plugin'],
		'rms': ['application/vnd.jcp.javame.midlet-rms', 'application/vnd.rn-realmedia', 'application/vnd.rn-realmedia-vbr'],
		'rmvb': ['application/vnd.rn-realmedia', 'application/vnd.rn-realmedia-vbr'],
		'rmx': ['application/vnd.rn-realmedia', 'application/vnd.rn-realmedia-vbr'],
		'rnc': ['application/relax-ng-compact-syntax', 'application/x-rnc'],
		'rng': ['application/xml', 'text/xml'],
		'roa': ['application/rpki-roa'],
		'roff': ['application/x-troff', 'text/troff', 'text/x-troff'],
		'ros': ['text/x-common-lisp'],
		'rp': ['image/vnd.rn-realpix'],
		'rp9': ['application/vnd.cloanto.rp9'],
		'rpm': ['application/x-redhat-package-manager', 'application/x-rpm'],
		'rpss': ['application/vnd.nokia.radio-presets'],
		'rpst': ['application/vnd.nokia.radio-preset'],
		'rq': ['application/sparql-query'],
		'rs': ['application/rls-services+xml', 'text/rust'],
		'rsat': ['application/atsc-rsat+xml'],
		'rsd': ['application/rsd+xml'],
		'rsheet': ['application/urc-ressheet+xml'],
		'rss': ['application/rss+xml', 'text/rss'],
		'rst': ['text/x-rst'],
		'rt': ['text/vnd.rn-realtext'],
		'rtf': ['application/rtf', 'text/rtf'],
		'rtx': ['text/richtext'],
		'run': ['application/x-makeself'],
		'rusd': ['application/route-usd+xml'],
		'rv': ['video/vnd.rn-realvideo', 'video/x-real-video'],
		'rvx': ['video/vnd.rn-realvideo', 'video/x-real-video'],
		'rw2': ['image/x-panasonic-raw2', 'image/x-panasonic-rw2'],
		'rz': ['application/x-rzip'],
		's': ['text/x-asm'],
		's3m': ['audio/s3m', 'audio/x-s3m'],
		'saf': ['application/vnd.yamaha.smaf-audio'],
		'sage': ['text/x-sagemath'],
		'sam': ['application/x-amipro'],
		'sami': ['application/x-sami'],
		'sap': ['application/x-sap-file', 'application/x-thomson-sap-image'],
		'sass': ['text/x-sass'],
		'sav': ['application/x-spss-sav', 'application/x-spss-savefile'],
		'sbml': ['application/sbml+xml'],
		'sc': ['application/vnd.ibm.secure-container', 'text/x-scala'],
		'scala': ['text/x-scala'],
		'scd': ['application/x-msschedule'],
		'scm': ['application/vnd.lotus-screencam', 'text/x-scheme'],
		'scn': ['application/x-godot-scene'],
		'scope': ['text/x-systemd-unit'],
		'scq': ['application/scvp-cv-request'],
		'scr': ['application/vnd.microsoft.portable-executable', 'application/x-ms-dos-executable', 'application/x-ms-ne-executable', 'application/x-msdownload'],
		'scs': ['application/scvp-cv-response'],
		'scss': ['text/x-scss'],
		'sct': ['image/x-sct'],
		'scurl': ['text/vnd.curl.scurl'],
		'sda': ['application/vnd.stardivision.draw', 'application/x-stardraw'],
		'sdc': ['application/vnd.stardivision.calc', 'application/x-starcalc'],
		'sdd': ['application/vnd.stardivision.impress', 'application/x-starimpress'],
		'sdkd': ['application/vnd.solent.sdkm+xml'],
		'sdkm': ['application/vnd.solent.sdkm+xml'],
		'sdm': ['application/vnd.stardivision.mail'],
		'sdp': ['application/sdp', 'application/vnd.sdp', 'application/vnd.stardivision.impress-packed', 'application/x-sdp'],
		'sds': ['application/vnd.stardivision.chart', 'application/x-starchart'],
		'sdw': ['application/vnd.stardivision.writer', 'application/x-starwriter'],
		'sea': ['application/x-sea'],
		'see': ['application/vnd.seemail'],
		'seed': ['application/vnd.fdsn.seed'],
		'sema': ['application/vnd.sema'],
		'semd': ['application/vnd.semd'],
		'semf': ['application/vnd.semf'],
		'senmlx': ['application/senml+xml'],
		'sensmlx': ['application/sensml+xml'],
		'ser': ['application/java-serialized-object'],
		'service': ['text/x-dbus-service', 'text/x-systemd-unit'],
		'setpay': ['application/set-payment-initiation'],
		'setreg': ['application/set-registration-initiation'],
		'sfc': ['application/vnd.nintendo.snes.rom', 'application/x-snes-rom'],
		'sfd-hdstx': ['application/vnd.hydrostatix.sof-data'],
		'sfs': ['application/vnd.spotfire.sfs', 'application/vnd.squashfs'],
		'sfv': ['text/x-sfv'],
		'sg': ['application/x-sg1000-rom'],
		'sgb': ['application/x-gameboy-rom'],
		'sgd': ['application/x-genesis-rom'],
		'sgf': ['application/x-go-sgf'],
		'sgi': ['image/sgi', 'image/x-sgi'],
		'sgl': ['application/vnd.stardivision.writer-global', 'application/x-starwriter-global'],
		'sgm': ['text/sgml'],
		'sgml': ['text/sgml'],
		'sh': ['application/x-sh', 'application/x-shellscript', 'text/x-sh'],
		'shape': ['application/x-dia-shape'],
		'shar': ['application/x-shar'],
		'shex': ['text/shex'],
		'shf': ['application/shf+xml'],
		'shn': ['application/x-shorten', 'audio/x-shorten'],
		'shtml': ['text/html'],
		'siag': ['application/x-siag'],
		'sid': ['audio/prs.sid', 'image/x-mrsid-image'],
		'sieve': ['application/sieve'],
		'sig': ['application/pgp-signature'],
		'sik': ['application/x-trash'],
		'sil': ['audio/silk'],
		'silo': ['model/mesh'],
		'sis': ['application/vnd.symbian.install'],
		'sisx': ['application/vnd.symbian.install', 'x-epoc/x-sisx-app'],
		'sit': ['application/x-stuffit', 'application/stuffit', 'application/x-sit'],
		'sitx': ['application/x-sitx', 'application/x-stuffitx'],
		'siv': ['application/sieve'],
		'sk': ['image/x-skencil'],
		'sk1': ['image/x-skencil'],
		'skd': ['application/vnd.koan'],
		'skm': ['application/vnd.koan'],
		'skp': ['application/vnd.koan'],
		'skr': ['application/pgp-keys'],
		'skt': ['application/vnd.koan'],
		'sldm': ['application/vnd.ms-powerpoint.slide.macroenabled.12'],
		'sldx': ['application/vnd.openxmlformats-officedocument.presentationml.slide'],
		'slice': ['text/x-systemd-unit'],
		'slim': ['text/slim'],
		'slk': ['application/x-sylk', 'text/spreadsheet'],
		'slm': ['text/slim'],
		'sls': ['application/route-s-tsid+xml'],
		'slt': ['application/vnd.epson.salt'],
		'sm': ['application/vnd.stepmania.stepchart'],
		'smaf': ['application/vnd.smaf', 'application/x-smaf'],
		'smc': ['application/vnd.nintendo.snes.rom', 'application/x-snes-rom'],
		'smd': ['application/x-genesis-rom', 'application/x-starmail'],
		'smf': ['application/vnd.stardivision.math', 'application/x-starmath'],
		'smi': ['application/smil', 'application/smil+xml', 'application/x-sami'],
		'smil': ['application/smil', 'application/smil+xml'],
		'smk': ['video/vnd.radgamettools.smacker'],
		'sml': ['application/smil', 'application/smil+xml'],
		'sms': ['application/x-sms-rom'],
		'smv': ['video/x-smv'],
		'smzip': ['application/vnd.stepmania.package'],
		'snap': ['application/vnd.snap'],
		'snd': ['audio/basic'],
		'snf': ['application/x-font-snf'],
		'so': ['application/x-sharedlib'],
		'socket': ['text/x-systemd-unit'],
		'spc': ['application/x-pkcs7-certificates'],
		'spd': ['application/x-font-speedo'],
		'spdx': ['text/spdx'],
		'spec': ['text/x-rpm-spec'],
		'spf': ['application/vnd.yamaha.smaf-phrase'],
		'spl': ['application/futuresplash', 'application/vnd.adobe.flash.movie', 'application/x-futuresplash', 'application/x-shockwave-flash'],
		'spm': ['application/x-source-rpm'],
		'spot': ['text/vnd.in3d.spot'],
		'spp': ['application/scvp-vp-response'],
		'spq': ['application/scvp-vp-request'],
		'spx': ['application/x-apple-systemprofiler+xml', 'audio/ogg', 'audio/x-speex', 'audio/x-speex+ogg'],
		'sqfs': ['application/vnd.squashfs'],
		'sql': ['application/sql', 'application/x-sql', 'text/x-sql'],
		'sqlite2': ['application/x-sqlite2'],
		'sqlite3': ['application/vnd.sqlite3', 'application/x-sqlite3'],
		'sqsh': ['application/vnd.squashfs'],
		'squashfs': ['application/vnd.squashfs'],
		'sr2': ['image/x-sony-sr2'],
		'src': ['application/x-wais-source'],
		'src.rpm': ['application/x-source-rpm'],
		'srf': ['image/x-sony-srf'],
		'srt': ['application/x-srt', 'application/x-subrip'],
		'sru': ['application/sru+xml'],
		'srx': ['application/sparql-results+xml'],
		'ss': ['text/x-scheme'],
		'ssa': ['text/x-ssa'],
		'ssdl': ['application/ssdl+xml'],
		'sse': ['application/vnd.kodak-descriptor'],
		'ssf': ['application/vnd.epson.ssf'],
		'ssml': ['application/ssml+xml'],
		'st': ['application/vnd.sailingtracker.track'],
		'stc': ['application/vnd.sun.xml.calc.template'],
		'std': ['application/vnd.sun.xml.draw.template'],
		'step': ['model/step'],
		'stf': ['application/vnd.wt.stf'],
		'sti': ['application/vnd.sun.xml.impress.template'],
		'stk': ['application/hyperstudio'],
		'stl': ['application/vnd.ms-pki.stl', 'model/stl', 'model/x.stl-ascii', 'model/x.stl-binary'],
		'stm': ['audio/x-stm'],
		'stp': ['model/step'],
		'stpx': ['model/step+xml'],
		'stpxz': ['model/step-xml+zip'],
		'stpz': ['model/step+zip'],
		'str': ['application/vnd.pg.format'],
		'stw': ['application/vnd.sun.xml.writer.template'],
		'sty': ['application/x-tex', 'text/x-tex'],
		'styl': ['text/stylus'],
		'stylus': ['text/stylus'],
		'sub': ['image/vnd.dvb.subtitle', 'text/vnd.dvb.subtitle', 'text/x-microdvd', 'text/x-mpsub', 'text/x-subviewer'],
		'sun': ['image/x-sun-raster'],
		'sus': ['application/vnd.sus-calendar'],
		'susp': ['application/vnd.sus-calendar'],
		'sv': ['text/x-svsrc'],
		'sv4cpio': ['application/x-sv4cpio'],
		'sv4crc': ['application/x-sv4crc'],
		'svc': ['application/vnd.dvb.service'],
		'svd': ['application/vnd.svd'],
		'svg': ['image/svg+xml', 'image/svg'],
		'svg.gz': ['image/svg+xml-compressed'],
		'svgz': ['image/svg+xml', 'image/svg+xml-compressed'],
		'svh': ['text/x-svhdr'],
		'swa': ['application/x-director'],
		'swap': ['text/x-systemd-unit'],
		'swf': ['application/futuresplash', 'application/vnd.adobe.flash.movie', 'application/x-shockwave-flash'],
		'swi': ['application/vnd.aristanetworks.swi'],
		'swidtag': ['application/swid+xml'],
		'swm': ['application/x-ms-wim'],
		'sxc': ['application/vnd.sun.xml.calc'],
		'sxd': ['application/vnd.sun.xml.draw'],
		'sxg': ['application/vnd.sun.xml.writer.global'],
		'sxi': ['application/vnd.sun.xml.impress'],
		'sxm': ['application/vnd.sun.xml.math'],
		'sxw': ['application/vnd.sun.xml.writer'],
		'sylk': ['application/x-sylk', 'text/spreadsheet'],
		'sys': ['application/vnd.microsoft.portable-executable'],
		't': ['application/x-perl', 'application/x-troff', 'text/troff', 'text/x-perl', 'text/x-troff'],
		't2t': ['text/x-txt2tags'],
		't3': ['application/x-t3vm-image'],
		't38': ['image/t38'],
		'taglet': ['application/vnd.mynfc'],
		'tak': ['audio/x-tak'],
		'tao': ['application/vnd.tao.intent-module-archive'],
		'tap': ['image/vnd.tencent.tap'],
		'tar': ['application/x-tar', 'application/x-gtar'],
		'tar.Z': ['application/x-tarz'],
		'tar.bz': ['application/x-bzip1-compressed-tar'],
		'tar.bz2': ['application/x-bzip-compressed-tar', 'application/x-bzip2-compressed-tar'],
		'tar.bz3': ['application/x-bzip3-compressed-tar'],
		'tar.gz': ['application/x-compressed-tar'],
		'tar.lrz': ['application/x-lrzip-compressed-tar'],
		'tar.lz': ['application/x-lzip-compressed-tar'],
		'tar.lz4': ['application/x-lz4-compressed-tar'],
		'tar.lzma': ['application/x-lzma-compressed-tar'],
		'tar.lzo': ['application/x-tzo'],
		'tar.rz': ['application/x-rzip-compressed-tar'],
		'tar.xz': ['application/x-xz-compressed-tar'],
		'tar.zst': ['application/x-zstd-compressed-tar'],
		'target': ['text/x-systemd-unit'],
		'taz': ['application/x-tarz'],
		'tb2': ['application/x-bzip-compressed-tar', 'application/x-bzip2-compressed-tar'],
		'tbz': ['application/x-bzip1-compressed-tar'],
		'tbz2': ['application/x-bzip-compressed-tar', 'application/x-bzip2-compressed-tar'],
		'tbz3': ['application/x-bzip3-compressed-tar'],
		'tcap': ['application/vnd.3gpp2.tcap'],
		'tcl': ['application/x-tcl', 'text/tcl', 'text/x-tcl'],
		'td': ['application/urc-targetdesc+xml'],
		'teacher': ['application/vnd.smart.teacher'],
		'tei': ['application/tei+xml'],
		'teicorpus': ['application/tei+xml'],
		'tex': ['application/x-tex', 'text/x-tex'],
		'texi': ['application/x-texinfo', 'text/x-texinfo'],
		'texinfo': ['application/x-texinfo', 'text/x-texinfo'],
		'text': ['text/plain'],
		'tfi': ['application/thraud+xml'],
		'tfm': ['application/x-tex-tfm'],
		'tfx': ['image/tiff-fx'],
		'tga': ['application/tga', 'application/x-targa', 'application/x-tga', 'image/targa', 'image/tga', 'image/x-icb', 'image/x-targa', 'image/x-tga'],
		'tgz': ['application/x-compressed-tar'],
		'theme': ['application/x-theme'],
		'themepack': ['application/x-windows-themepack'],
		'thmx': ['application/vnd.ms-officetheme'],
		'tif': ['image/tiff'],
		'tiff': ['image/tiff'],
		'timer': ['text/x-systemd-unit'],
		'tk': ['application/x-tcl', 'text/tcl', 'text/x-tcl'],
		'tlrz': ['application/x-lrzip-compressed-tar'],
		'tlz': ['application/x-lzma-compressed-tar'],
		'tmo': ['application/vnd.tmobile-livetv'],
		'tmx': ['application/x-tiled-tmx'],
		'tnef': ['application/ms-tnef', 'application/vnd.ms-tnef'],
		'tnf': ['application/ms-tnef', 'application/vnd.ms-tnef'],
		'toc': ['application/x-cdrdao-toc'],
		'toml': ['application/toml'],
		'torrent': ['application/x-bittorrent'],
		'tpic': ['application/tga', 'application/x-targa', 'application/x-tga', 'image/targa', 'image/tga', 'image/x-icb', 'image/x-targa', 'image/x-tga'],
		'tpl': ['application/vnd.groove-tool-template'],
		'tpt': ['application/vnd.trid.tpt'],
		'tr': ['application/x-troff', 'text/troff', 'text/x-troff'],
		'tra': ['application/vnd.trueapp'],
		'tres': ['application/x-godot-resource'],
		'trig': ['application/trig', 'application/x-trig'],
		'trm': ['application/x-msterminal'],
		'trz': ['application/x-rzip-compressed-tar'],
		'ts': ['application/x-linguist', 'text/vnd.qt.linguist', 'text/vnd.trolltech.linguist', 'video/mp2t'],
		'tscn': ['application/x-godot-scene'],
		'tsd': ['application/timestamped-data'],
		'tsv': ['text/tab-separated-values'],
		'tsx': ['application/x-tiled-tsx'],
		'tta': ['audio/tta', 'audio/x-tta'],
		'ttc': ['font/collection'],
		'ttf': ['application/x-font-truetype', 'application/x-font-ttf', 'font/ttf'],
		'ttl': ['text/turtle'],
		'ttml': ['application/ttml+xml'],
		'ttx': ['application/x-font-ttx'],
		'twd': ['application/vnd.simtech-mindmapper'],
		'twds': ['application/vnd.simtech-mindmapper'],
		'twig': ['text/x-twig'],
		'txd': ['application/vnd.genomatix.tuxedo'],
		'txf': ['application/vnd.mobius.txf'],
		'txt': ['text/plain'],
		'txz': ['application/x-xz-compressed-tar'],
		'typ': ['text/x-typst'],
		'tzo': ['application/x-tzo'],
		'tzst': ['application/x-zstd-compressed-tar'],
		'u32': ['application/x-authorware-bin'],
		'u3d': ['model/u3d'],
		'u8dsn': ['message/global-delivery-status'],
		'u8hdr': ['message/global-headers'],
		'u8mdn': ['message/global-disposition-notification'],
		'u8msg': ['message/global'],
		'ubj': ['application/ubjson'],
		'udeb': ['application/vnd.debian.binary-package', 'application/x-deb', 'application/x-debian-package'],
		'ufd': ['application/vnd.ufdl'],
		'ufdl': ['application/vnd.ufdl'],
		'ufraw': ['application/x-ufraw'],
		'ui': ['application/x-designer', 'application/x-gtk-builder'],
		'uil': ['text/x-uil'],
		'ult': ['audio/x-mod'],
		'ulx': ['application/x-glulx'],
		'umj': ['application/vnd.umajin'],
		'unf': ['application/x-nes-rom'],
		'uni': ['audio/x-mod'],
		'unif': ['application/x-nes-rom'],
		'unityweb': ['application/vnd.unity'],
		'uo': ['application/vnd.uoml+xml'],
		'uoml': ['application/vnd.uoml+xml'],
		'uri': ['text/uri-list'],
		'uris': ['text/uri-list'],
		'url': ['application/x-mswinurl'],
		'urls': ['text/uri-list'],
		'usda': ['model/vnd.usda'],
		'usdz': ['model/vnd.usdz+zip'],
		'ustar': ['application/x-ustar'],
		'utz': ['application/vnd.uiq.theme'],
		'uu': ['text/x-uuencode'],
		'uue': ['text/x-uuencode', 'zz-application/zz-winassoc-uu'],
		'uva': ['audio/vnd.dece.audio'],
		'uvd': ['application/vnd.dece.data'],
		'uvf': ['application/vnd.dece.data'],
		'uvg': ['image/vnd.dece.graphic'],
		'uvh': ['video/vnd.dece.hd'],
		'uvi': ['image/vnd.dece.graphic'],
		'uvm': ['video/vnd.dece.mobile'],
		'uvp': ['video/vnd.dece.pd'],
		'uvs': ['video/vnd.dece.sd'],
		'uvt': ['application/vnd.dece.ttml+xml'],
		'uvu': ['video/vnd.uvvu.mp4'],
		'uvv': ['video/vnd.dece.video'],
		'uvva': ['audio/vnd.dece.audio'],
		'uvvd': ['application/vnd.dece.data'],
		'uvvf': ['application/vnd.dece.data'],
		'uvvg': ['image/vnd.dece.graphic'],
		'uvvh': ['video/vnd.dece.hd'],
		'uvvi': ['image/vnd.dece.graphic'],
		'uvvm': ['video/vnd.dece.mobile'],
		'uvvp': ['video/vnd.dece.pd'],
		'uvvs': ['video/vnd.dece.sd'],
		'uvvt': ['application/vnd.dece.ttml+xml'],
		'uvvu': ['video/vnd.uvvu.mp4'],
		'uvvv': ['video/vnd.dece.video'],
		'uvvx': ['application/vnd.dece.unspecified'],
		'uvvz': ['application/vnd.dece.zip'],
		'uvx': ['application/vnd.dece.unspecified'],
		'uvz': ['application/vnd.dece.zip'],
		'v': ['text/x-verilog'],
		'v64': ['application/x-n64-rom'],
		'vala': ['text/x-vala'],
		'vapi': ['text/x-vala'],
		'vb': ['application/x-virtual-boy-rom', 'text/x-vb'],
		'vbe': ['text/vbscript.encode'],
		'vbox': ['application/x-virtualbox-vbox'],
		'vbox-extpack': ['application/x-virtualbox-vbox-extpack'],
		'vbs': ['text/vbs', 'text/vbscript'],
		'vcard': ['text/directory', 'text/vcard', 'text/x-vcard'],
		'vcd': ['application/x-cdlink'],
		'vcf': ['text/x-vcard', 'text/directory', 'text/vcard'],
		'vcg': ['application/vnd.groove-vcard'],
		'vcs': ['application/ics', 'text/calendar', 'text/x-vcalendar'],
		'vct': ['text/directory', 'text/vcard', 'text/x-vcard'],
		'vcx': ['application/vnd.vcx'],
		'vda': ['application/tga', 'application/x-targa', 'application/x-tga', 'image/targa', 'image/tga', 'image/x-icb', 'image/x-targa', 'image/x-tga'],
		'vdi': ['application/x-vdi-disk', 'application/x-virtualbox-vdi'],
		'vds': ['model/vnd.sap.vds'],
		'vhd': ['application/x-vhd-disk', 'application/x-virtualbox-vhd', 'text/x-vhdl'],
		'vhdl': ['text/x-vhdl'],
		'vhdx': ['application/x-vhdx-disk', 'application/x-virtualbox-vhdx'],
		'vis': ['application/vnd.visionary'],
		'viv': ['video/vivo', 'video/vnd.vivo'],
		'vivo': ['video/vivo', 'video/vnd.vivo'],
		'vlc': ['application/m3u', 'audio/m3u', 'audio/mpegurl', 'audio/x-m3u', 'audio/x-mp3-playlist', 'audio/x-mpegurl'],
		'vmdk': ['application/x-virtualbox-vmdk', 'application/x-vmdk-disk'],
		'vob': ['video/mpeg', 'video/mpeg-system', 'video/x-mpeg', 'video/x-mpeg-system', 'video/x-mpeg2', 'video/x-ms-vob'],
		'voc': ['audio/x-voc'],
		'vor': ['application/vnd.stardivision.writer', 'application/x-starwriter'],
		'vox': ['application/x-authorware-bin'],
		'vpc': ['application/x-vhd-disk', 'application/x-virtualbox-vhd'],
		'vrm': ['model/vrml'],
		'vrml': ['model/vrml'],
		'vsd': ['application/vnd.visio'],
		'vsdm': ['application/vnd.ms-visio.drawing.macroenabled.main+xml'],
		'vsdx': ['application/vnd.ms-visio.drawing.main+xml'],
		'vsf': ['application/vnd.vsf'],
		'vss': ['application/vnd.visio'],
		'vssm': ['application/vnd.ms-visio.stencil.macroenabled.main+xml'],
		'vssx': ['application/vnd.ms-visio.stencil.main+xml'],
		'vst': ['application/tga', 'application/vnd.visio', 'application/x-targa', 'application/x-tga', 'image/targa', 'image/tga', 'image/x-icb', 'image/x-targa', 'image/x-tga'],
		'vstm': ['application/vnd.ms-visio.template.macroenabled.main+xml'],
		'vstx': ['application/vnd.ms-visio.template.main+xml'],
		'vsw': ['application/vnd.visio'],
		'vtf': ['image/vnd.valve.source.texture'],
		'vtt': ['text/vtt'],
		'vtu': ['model/vnd.vtu'],
		'vxml': ['application/voicexml+xml'],
		'w3d': ['application/x-director'],
		'wad': ['application/x-doom', 'application/x-doom-wad', 'application/x-wii-wad'],
		'wadl': ['application/vnd.sun.wadl+xml'],
		'war': ['application/java-archive'],
		'wasm': ['application/wasm'],
		'wav': ['audio/wav', 'audio/vnd.wave', 'audio/wave', 'audio/x-wav'],
		'wax': ['application/x-ms-asx', 'audio/x-ms-asx', 'audio/x-ms-wax', 'video/x-ms-wax', 'video/x-ms-wmx', 'video/x-ms-wvx'],
		'wb1': ['application/x-quattropro'],
		'wb2': ['application/x-quattropro'],
		'wb3': ['application/x-quattropro'],
		'wbmp': ['image/vnd.wap.wbmp'],
		'wbs': ['application/vnd.criticaltools.wbs+xml'],
		'wbxml': ['application/vnd.wap.wbxml'],
		'wcm': ['application/vnd.ms-works'],
		'wdb': ['application/vnd.ms-works'],
		'wdp': ['image/jxr', 'image/vnd.ms-photo'],
		'weba': ['audio/webm'],
		'webapp': ['application/x-web-app-manifest+json'],
		'webm': ['video/webm'],
		'webmanifest': ['application/manifest+json'],
		'webp': ['image/webp'],
		'wg': ['application/vnd.pmi.widget'],
		'wgsl': ['text/wgsl'],
		'wgt': ['application/widget'],
		'wif': ['application/watcherinfo+xml'],
		'wim': ['application/x-ms-wim'],
		'wk1': ['application/lotus123', 'application/vnd.lotus-1-2-3', 'application/wk1', 'application/x-123', 'application/x-lotus123', 'zz-application/zz-winassoc-123'],
		'wk3': ['application/lotus123', 'application/vnd.lotus-1-2-3', 'application/wk1', 'application/x-123', 'application/x-lotus123', 'zz-application/zz-winassoc-123'],
		'wk4': ['application/lotus123', 'application/vnd.lotus-1-2-3', 'application/wk1', 'application/x-123', 'application/x-lotus123', 'zz-application/zz-winassoc-123'],
		'wkdownload': ['application/x-partial-download'],
		'wks': ['application/lotus123', 'application/vnd.lotus-1-2-3', 'application/vnd.ms-works', 'application/wk1', 'application/x-123', 'application/x-lotus123', 'zz-application/zz-winassoc-123'],
		'wm': ['video/x-ms-wm'],
		'wma': ['audio/x-ms-wma', 'audio/wma'],
		'wmd': ['application/x-ms-wmd'],
		'wmf': ['application/wmf', 'application/x-msmetafile', 'application/x-wmf', 'image/wmf', 'image/x-win-metafile', 'image/x-wmf'],
		'wml': ['text/vnd.wap.wml'],
		'wmlc': ['application/vnd.wap.wmlc'],
		'wmls': ['text/vnd.wap.wmlscript'],
		'wmlsc': ['application/vnd.wap.wmlscriptc'],
		'wmv': ['audio/x-ms-wmv', 'video/x-ms-wmv'],
		'wmx': ['application/x-ms-asx', 'audio/x-ms-asx', 'video/x-ms-wax', 'video/x-ms-wmx', 'video/x-ms-wvx'],
		'wmz': ['application/x-ms-wmz', 'application/x-msmetafile'],
		'woff': ['application/font-woff', 'application/x-font-woff', 'font/woff'],
		'woff2': ['font/woff2'],
		'wp': ['application/vnd.wordperfect', 'application/wordperfect', 'application/x-wordperfect'],
		'wp4': ['application/vnd.wordperfect', 'application/wordperfect', 'application/x-wordperfect'],
		'wp5': ['application/vnd.wordperfect', 'application/wordperfect', 'application/x-wordperfect'],
		'wp6': ['application/vnd.wordperfect', 'application/wordperfect', 'application/x-wordperfect'],
		'wpd': ['application/vnd.wordperfect', 'application/wordperfect', 'application/x-wordperfect'],
		'wpg': ['application/x-wpg'],
		'wpl': ['application/vnd.ms-wpl'],
		'wpp': ['application/vnd.wordperfect', 'application/wordperfect', 'application/x-wordperfect'],
		'wps': ['application/vnd.ms-works'],
		'wqd': ['application/vnd.wqd'],
		'wri': ['application/x-mswrite'],
		'wrl': ['model/vrml'],
		'ws': ['application/x-wonderswan-rom'],
		'wsc': ['application/x-wonderswan-color-rom', 'message/vnd.wfa.wsc'],
		'wsdl': ['application/wsdl+xml'],
		'wsgi': ['text/x-python'],
		'wspolicy': ['application/wspolicy+xml'],
		'wtb': ['application/vnd.webturbo'],
		'wv': ['audio/x-wavpack'],
		'wvc': ['audio/x-wavpack-correction'],
		'wvp': ['audio/x-wavpack'],
		'wvx': ['application/x-ms-asx', 'audio/x-ms-asx', 'video/x-ms-wax', 'video/x-ms-wmx', 'video/x-ms-wvx'],
		'wwf': ['application/wwf', 'application/x-wwf'],
		'x32': ['application/x-authorware-bin'],
		'x3d': ['model/x3d+xml'],
		'x3db': ['model/x3d+binary', 'model/x3d+fastinfoset'],
		'x3dbz': ['model/x3d+binary'],
		'x3dv': ['model/x3d+vrml', 'model/x3d-vrml'],
		'x3dvz': ['model/x3d+vrml'],
		'x3dz': ['model/x3d+xml'],
		'x3f': ['image/x-sigma-x3f'],
		'x_b': ['model/vnd.parasolid.transmit.binary'],
		'x_t': ['model/vnd.parasolid.transmit.text'],
		'xac': ['application/x-gnucash'],
		'xaml': ['application/xaml+xml'],
		'xap': ['application/x-silverlight-app'],
		'xar': ['application/vnd.xara', 'application/x-xar'],
		'xav': ['application/xcap-att+xml'],
		'xbap': ['application/x-ms-xbap'],
		'xbd': ['application/vnd.fujixerox.docuworks.binder'],
		'xbel': ['application/x-xbel'],
		'xbl': ['application/xml', 'text/xml'],
		'xbm': ['image/x-xbitmap'],
		'xca': ['application/xcap-caps+xml'],
		'xcf': ['image/x-xcf'],
		'xcf.bz2': ['image/x-compressed-xcf'],
		'xcf.gz': ['image/x-compressed-xcf'],
		'xci': ['application/x-nintendo-switch-xci', 'application/x-nx-xci'],
		'xcs': ['application/calendar+xml'],
		'xdcf': ['application/vnd.gov.sk.xmldatacontainer+xml'],
		'xdf': ['application/mrb-consumer+xml', 'application/mrb-publish+xml', 'application/xcap-diff+xml'],
		'xdgapp': ['application/vnd.flatpak', 'application/vnd.xdgapp'],
		'xdm': ['application/vnd.syncml.dm+xml'],
		'xdp': ['application/vnd.adobe.xdp+xml'],
		'xdssc': ['application/dssc+xml'],
		'xdw': ['application/vnd.fujixerox.docuworks'],
		'xel': ['application/xcap-el+xml'],
		'xenc': ['application/xenc+xml'],
		'xer': ['application/patch-ops-error+xml', 'application/xcap-error+xml'],
		'xfdf': ['application/vnd.adobe.xfdf', 'application/xfdf'],
		'xfdl': ['application/vnd.xfdl'],
		'xhe': ['audio/usac'],
		'xht': ['application/xhtml+xml'],
		'xhtm': ['application/vnd.pwg-xhtml-print+xml'],
		'xhtml': ['application/xhtml+xml'],
		'xhvml': ['application/xv+xml'],
		'xi': ['audio/x-xi'],
		'xif': ['image/vnd.xiff'],
		'xla': ['application/msexcel', 'application/vnd.ms-excel', 'application/x-msexcel', 'zz-application/zz-winassoc-xls'],
		'xlam': ['application/vnd.ms-excel.addin.macroenabled.12'],
		'xlc': ['application/msexcel', 'application/vnd.ms-excel', 'application/x-msexcel', 'zz-application/zz-winassoc-xls'],
		'xld': ['application/msexcel', 'application/vnd.ms-excel', 'application/x-msexcel', 'zz-application/zz-winassoc-xls'],
		'xlf': ['application/x-xliff', 'application/x-xliff+xml', 'application/xliff+xml'],
		'xliff': ['application/x-xliff', 'application/xliff+xml'],
		'xll': ['application/msexcel', 'application/vnd.ms-excel', 'application/x-msexcel', 'zz-application/zz-winassoc-xls'],
		'xlm': ['application/msexcel', 'application/vnd.ms-excel', 'application/x-msexcel', 'zz-application/zz-winassoc-xls'],
		'xlr': ['application/vnd.ms-works'],
		'xls': ['application/vnd.ms-excel', 'application/msexcel', 'application/x-msexcel', 'zz-application/zz-winassoc-xls'],
		'xlsb': ['application/vnd.ms-excel.sheet.binary.macroenabled.12'],
		'xlsm': ['application/vnd.ms-excel.sheet.macroenabled.12'],
		'xlsx': ['application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'],
		'xlt': ['application/msexcel', 'application/vnd.ms-excel', 'application/x-msexcel', 'zz-application/zz-winassoc-xls'],
		'xltm': ['application/vnd.ms-excel.template.macroenabled.12'],
		'xltx': ['application/vnd.openxmlformats-officedocument.spreadsheetml.template'],
		'xlw': ['application/msexcel', 'application/vnd.ms-excel', 'application/x-msexcel', 'zz-application/zz-winassoc-xls'],
		'xm': ['audio/x-xm', 'audio/xm'],
		'xmf': ['audio/x-xmf', 'audio/xmf'],
		'xmi': ['text/x-xmi'],
		'xml': ['application/xml', 'text/xml'],
		'xns': ['application/xcap-ns+xml'],
		'xo': ['application/vnd.olpc-sugar'],
		'xop': ['application/xop+xml'],
		'xpi': ['application/x-xpinstall'],
		'xpl': ['application/xproc+xml'],
		'xpm': ['image/x-xpixmap', 'image/x-xpm'],
		'xpr': ['application/vnd.is-xpr'],
		'xps': ['application/vnd.ms-xpsdocument', 'application/xps'],
		'xpw': ['application/vnd.intercon.formnet'],
		'xpx': ['application/vnd.intercon.formnet'],
		'xsd': ['application/xml', 'text/xml'],
		'xsf': ['application/prs.xsf+xml'],
		'xsl': ['application/xml', 'application/xslt+xml'],
		'xslfo': ['text/x-xslfo'],
		'xslt': ['application/xslt+xml'],
		'xsm': ['application/vnd.syncml+xml'],
		'xspf': ['application/x-xspf+xml', 'application/xspf+xml'],
		'xul': ['application/vnd.mozilla.xul+xml'],
		'xvm': ['application/xv+xml'],
		'xvml': ['application/xv+xml'],
		'xwd': ['image/x-xwindowdump'],
		'xyz': ['chemical/x-xyz'],
		'xz': ['application/x-xz'],
		'yaml': ['application/yaml', 'application/x-yaml', 'text/x-yaml', 'text/yaml'],
		'yang': ['application/yang'],
		'yin': ['application/yin+xml'],
		'yml': ['application/yaml', 'application/x-yaml', 'text/x-yaml', 'text/yaml'],
		'ymp': ['text/x-suse-ymp'],
		'yt': ['application/vnd.youtube.yt', 'video/vnd.youtube.yt'],
		'z1': ['application/x-zmachine'],
		'z2': ['application/x-zmachine'],
		'z3': ['application/x-zmachine'],
		'z4': ['application/x-zmachine'],
		'z5': ['application/x-zmachine'],
		'z6': ['application/x-zmachine'],
		'z64': ['application/x-n64-rom'],
		'z7': ['application/x-zmachine'],
		'z8': ['application/x-zmachine'],
		'zabw': ['application/x-abiword'],
		'zaz': ['application/vnd.zzazz.deck+xml'],
		'zim': ['application/x-openzim'],
		'zip': ['application/zip', 'application/x-zip', 'application/x-zip-compressed'],
		'zipx': ['application/x-zip', 'application/x-zip-compressed', 'application/zip'],
		'zir': ['application/vnd.zul'],
		'zirz': ['application/vnd.zul'],
		'zmm': ['application/vnd.handheld-entertainment+xml'],
		'zoo': ['application/x-zoo'],
		'zpaq': ['application/x-zpaq'],
		'zsav': ['application/x-spss-sav', 'application/x-spss-savefile'],
		'zst': ['application/zstd'],
		'zz': ['application/zlib']
	};
	
	extension = extension.toLowerCase().replace(/^\./, '');
	return mimeTypes[extension]?.[0] || 'application/octet-stream';
}
